package nl.tudelft.pogamut.unreal.agent.module.shooting.util;

import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.base3d.worldview.object.Rotation;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;

/**
 * Utilities to determine if a player is facing some location.
 * 
 * @author mpkorstanje
 * 
 */
public class FacingUtil {

	/**
	 * What angle is considered to be maximum facing angle by default (in
	 * degrees).
	 */
	public static final double IS_FACING_ANGLE = 6;

	/**
	 * Tells if the player is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * 
	 * @param location
	 *            input location.
	 * @return True, if the bot is facing input location.
	 */
	public static boolean isFacing(AgentInfo info, ILocated location) {
		return isFacing(info, location, IS_FACING_ANGLE);
	}

	/**
	 * Tells if the agent is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * @param location
	 *            input location.
	 * @param angle
	 *            specifies maximum angle (in degrees) that will be still
	 *            considered as facing angle.
	 * @return True, if the angle between agent facing vector and input location
	 *         is smaller or equal to input angle.
	 */
	public static boolean isFacing(AgentInfo info, ILocated location, double angle) {
		if (info == null || location == null)
			return false;

		return isFacing(info.getLocation(), info.getRotation(), location.getLocation(), angle);
	}

	/**
	 * Tells you if the rotation at the origin is facing the target location.
	 * 
	 * @param origin
	 *            the location of the object that should be facing
	 * @param rotation
	 *            rotation of the object that should be facing
	 * @param target
	 *            what it should be facing
	 * @param angle
	 *            specifies maximum angle (in degrees) that will be still
	 *            considered as facing angle.
	 * @return True, if the angle between direction vector and input location is
	 *         smaller or equal to input angle.
	 */
	public static boolean isFacing(Location origin, Rotation rotation, Location target, double angle) {
		if (origin == null || rotation == null || target == null)
			return false;

		Location directionVector = target.sub(origin).getNormalized();
		Location agentFaceVector = rotation.toLocation().getNormalized();

		if (Math.acos(directionVector.dot(agentFaceVector)) <= Math.toRadians(angle))
			return true;

		return false;

	}

	/**
	 * Tells if the player is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * 
	 * @param location
	 *            input location.
	 * @return True, if the bot is facing input location.
	 */
	public static boolean isFacing(Player player, ILocated location) {
		return isFacing(player, location, IS_FACING_ANGLE);
	}

	/**
	 * Tells if the player is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * @param location
	 *            input location.
	 * @param angle
	 *            specifies maximum angle (in degrees) that will be still
	 *            considered as facing angle.
	 * @return True, if the angle between agent facing vector and input location
	 *         is smaller or equal to input angle.
	 */
	public static boolean isFacing(Player player, ILocated location, double angle) {
		if (player == null || location == null)
			return false;

		return isFacing(player.getLocation(), player.getRotation(), location.getLocation(), angle);
	}

	/**
	 * Tells if the player is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * 
	 * @param location
	 *            input location.
	 * @return True, if the bot is facing input location.
	 */
	public static boolean isFacing2D(AgentInfo info, ILocated location) {
		return isFacing2D(info, location, IS_FACING_ANGLE);
	}

	/**
	 * Tells if the agent is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * @param location
	 *            input location.
	 * @param angle
	 *            specifies maximum angle (in degrees) that will be still
	 *            considered as facing angle.
	 * @return True, if the angle between agent facing vector and input location
	 *         is smaller or equal to input angle.
	 */
	public static boolean isFacing2D(AgentInfo info, ILocated location, double angle) {
		if (info == null || location == null)
			return false;

		return isFacing2D(info.getLocation(), info.getRotation(), location.getLocation(), angle);
	}

	/**
	 * Tells you if the rotation at the origin is facing the target location in
	 * the XY (horizontal) plane.
	 * 
	 * @param origin
	 *            the location of the object that should be facing
	 * @param rotation
	 *            rotation of the object that should be facing
	 * @param target
	 *            what it should be facing
	 * @param angle
	 *            specifies maximum angle (in degrees) that will be still
	 *            considered as facing angle.
	 * @return True, if the angle between direction vector and input location is
	 *         smaller or equal to input angle.
	 */
	public static boolean isFacing2D(Location origin, Rotation rotation, Location target, double angle) {
		if (origin == null || rotation == null || target == null)
			return false;

		Location directionVector = target.sub(origin).setZ(0).getNormalized();
		Location agentFaceVector = rotation.toLocation().setZ(0).getNormalized();

		if (Math.acos(directionVector.dot(agentFaceVector)) <= Math.toRadians(angle))
			return true;

		return false;

	}

	/**
	 * Tells if the player is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * 
	 * @param location
	 *            input location.
	 * @return True, if the bot is facing input location.
	 */
	public static boolean isFacing2D(Player player, ILocated location) {
		return isFacing2D(player, location, IS_FACING_ANGLE);
	}

	/**
	 * Tells if the player is currently facing input location.
	 * 
	 * @param player
	 *            the player:
	 * @param location
	 *            input location.
	 * @param angle
	 *            specifies maximum angle (in degrees) that will be still
	 *            considered as facing angle.
	 * @return True, if the angle between agent facing vector and input location
	 *         is smaller or equal to input angle.
	 */
	public static boolean isFacing2D(Player player, ILocated location, double angle) {
		if (player == null || location == null)
			return false;

		return isFacing2D(player.getLocation(), player.getRotation(), location.getLocation(), angle);
	}

}
