/**
 * BaseUnrealEnvironment, an implementation of the environment interface standard that 
 * facilitates the connection between GOAL and the UT2004 engine. 
 * 
 * Copyright (C) 2012 BaseUnrealEnvironment authors.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.tudelft.goal.unreal.messages;

import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Logger;

import nl.tudelft.goal.EIS2Java.exception.TranslationException;
import nl.tudelft.goal.unreal.environment.UnrealEnvironmentException;
import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IAgentParameters;
import cz.cuni.amis.pogamut.base.communication.connection.IWorldConnectionAddress;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.component.IComponent;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.ut2004.bot.params.UT2004BotParameters;
import eis.iilang.Parameter;

/**
 * Base class for the {@link EnvironmentParameters} and {@link BotParameters}.
 * These parameters are instantiated using the init values provided by EIS.
 * 
 * @author mpkorstanje
 * 
 */
public abstract class Parameters extends UT2004BotParameters implements IComponent {

	public static final int BOT_SERVER_PORT = 3000;
	public static final int CONTROL_SERVER_PORT = 3001;
	public static final String LOCAL_HOST = "127.0.0.1";
	public static final String DEFAULT_NAME = "UnrealGoalBot";

	protected final Logger log;

	public Parameters(IAgentLogger log) {
		super();
		this.log = log.getCategory(this);
	}

	@Override
	public void assignDefaults(IAgentParameters defaults) {
		super.assignDefaults(defaults);
	}

	public Parameters(Map<String, Parameter> parameters, IAgentLogger log) throws UnrealEnvironmentException {
		this(log);

		for (Entry<String, Parameter> entry : parameters.entrySet()) {
			String key = entry.getKey();
			Parameter value = entry.getValue();
			try {
				setKey(Key.parseKey(key), value);
			} catch (TranslationException e) {
				String message = String.format("Could not set %s to %s. Cause: %s", key, value, e.getMessage());
				throw new UnrealEnvironmentException(message, e);
			} catch (IllegalArgumentException e) {
				String message = String.format("Could not set %s to %s. Cause: %s", key, value, e.getMessage());
				throw new UnrealEnvironmentException(message, e);
			}
		}
	}

	protected abstract void setKey(Key key, Parameter value) throws TranslationException;

	@Override
	public Parameters setAgentId(IAgentId agentId) {
		super.setAgentId(agentId);
		return this;
	}

	public Parameters setAgentId(String name) {
		assert name != null;
		this.setAgentId(new AgentId(name));
		return this;
	}

	@Override
	public Parameters setWorldAddress(IWorldConnectionAddress address) {
		super.setWorldAddress(address);
		log.info(String.format("Set %s to %s.", Key.BOTSERVER, address));
		return this;
	}

	public Parameters setWorldAddress(String host, Integer port) {
		this.setWorldAddress(new SocketConnectionAddress(host, port));
		return this;
	}

}