/**
 * BaseUnrealEnvironment, an implementation of the environment interface standard that 
 * facilitates the connection between GOAL and the UT2004 engine. 
 * 
 * Copyright (C) 2012 BaseUnrealEnvironment authors.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.tudelft.goal.unreal.messages;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import nl.tudelft.goal.unreal.environment.UnrealEnvironmentException;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IAgentParameters;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.ISocketConnectionAddress;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.component.IComponent;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.utils.token.IToken;
import cz.cuni.amis.utils.token.Tokens;
import eis.eis2java.exception.TranslationException;
import eis.eis2java.translation.Translator;
import eis.iilang.Identifier;
import eis.iilang.Parameter;

/**
 * Holds parameters specific for the environment.
 * 
 * Parameters stored are:
 * <ul>
 * <li>{@link Key#CONTROLSERVER}</li>
 * <li>{@link Key#LOGLEVEL}</li>
 * <li>{@link Key#CONTROLSERVERNAME}</li>
 * <li>{@link Key#VISUALIZERSERVER}</li>
 * </ul>
 * 
 * Also provides functionality to assign defaults to parameters that have not
 * been assigned.
 * 
 * @author M.P. Korstanje
 * 
 */
public final class EnvironmentParameters extends Parameters implements IComponent {

	// Environment parameters
	private Level logLevel;
	private List<String> botNames;
	private URI visualizerServer;

	public EnvironmentParameters(IAgentLogger logger) {
		super(logger);
	}

	public EnvironmentParameters(Map<String, Parameter> parameters, IAgentLogger logger) throws UnrealEnvironmentException {
		super(parameters, logger);
	}

	@Override
	public void assignDefaults(IAgentParameters defaults) {
		super.assignDefaults(defaults);

		if (defaults instanceof EnvironmentParameters) {
			EnvironmentParameters parameters = (EnvironmentParameters) defaults;

			if (logLevel == null)
				logLevel = parameters.getLogLevel();
			if (botNames == null)
				botNames = new ArrayList<String>(parameters.getBotNames());
			if (visualizerServer == null)
				visualizerServer = parameters.getVisualizerServer();
		}
	}

	@Override
	public IToken getComponentId() {
		return Tokens.get(getClass().getSimpleName());
	}

	public Level getLogLevel() {
		return logLevel;
	}

	public List<String> getBotNames() {
		return botNames;
	}

	public void setVisualizerServer(URI uri) {
		this.visualizerServer = uri;
	}

	public URI getVisualizerServer() {
		return visualizerServer;
	}

	public URI getUTServer() {
		SocketConnectionAddress address = (SocketConnectionAddress) getWorldAddress();

		if (address.getHost() == null || address.getPort() == -1)
			return null;

		try {
			URI uri = new URI("ut://" + address.getHost() + ":" + address.getPort());
			return uri;
		} catch (URISyntaxException e) {
			e.printStackTrace();
		}

		return null;
	}

	/**
	 * Returns a map of this parameter object.
	 * 
	 * 
	 * @return
	 */
	public Map<String, Parameter> map() {

		Map<String, Parameter> map = new HashMap<String, Parameter>();

		if (logLevel != null) {
			map.put(Key.LOGLEVEL.toString(), new Identifier(logLevel.toString()));
		}
		if (getAgentId() != null) {
			map.put(Key.CONTROLSERVERNAME.toString(), new Identifier(getAgentId().getName().getFlag()));
		}
		if (getWorldAddress() != null) {
			assert getWorldAddress() instanceof ISocketConnectionAddress;
			ISocketConnectionAddress address = (ISocketConnectionAddress) getWorldAddress();
			String adressString = "ut://" + address.getHost() + ":" + address.getPort();
			map.put(Key.CONTROLSERVER.toString(), new Identifier(adressString));
		}
		assert map.size() + 2 == Key.values().length : "Missing values.";

		return map;

	}

	public Parameters setLogLevel(Level level) {
		assert level != null;
		log.info(String.format("Set %s to %s.", Key.LOGLEVEL, level));
		this.logLevel = level;
		return this;
	}

	public Parameters setBotNames(List<String> botNames) {
		assert botNames != null;

		this.botNames = new ArrayList<String>();
		this.botNames.addAll(botNames);

		log.info(String.format("Set %s to %s.", Key.BOTNAMES, this.botNames));
		return this;
	}


	@Override
	protected void setKey(Key key, Parameter value) throws TranslationException {

		switch (key) {
		case LOGLEVEL:
			setLogLevel(value);
			break;
		case CONTROLSERVER:
			setWorldAddress(value);
			break;
		case VISUALIZERSERVER:
			setVisualizerServer(value);
			break;
		case CONTROLSERVERNAME:
			setServerName(value);
			break;
		case BOTNAMES:
			setBotNames(value);
			break;
		default:
			// Not one of our keys.
			break;
		}

	}

	private void setLogLevel(Parameter value) throws TranslationException {
		setLogLevel(Translator.getInstance().translate2Java(value, Level.class));
	}

	private void setServerName(Parameter value) throws TranslationException {
		setAgentId(Translator.getInstance().translate2Java(value, AgentId.class));
	}

	private void setVisualizerServer(Parameter value) throws TranslationException {
		URI uri = Translator.getInstance().translate2Java(value, URI.class);
		setVisualizerServer(uri);
	}

	private void setBotNames(Parameter value) throws TranslationException {
		setBotNames(Translator.getInstance().translate2Java(value, StringList.class));
	}

	private void setWorldAddress(Parameter value) throws TranslationException {
		URI uri = Translator.getInstance().translate2Java(value, URI.class);
		setWorldAddress(new SocketConnectionAddress(uri));
	}

	public static EnvironmentParameters getDefaults(IAgentLogger log) {

		EnvironmentParameters parameters = new EnvironmentParameters(log);

		parameters.logLevel = Level.WARNING;
		parameters.botNames = new ArrayList<String>();
		parameters.visualizerServer = null;
		parameters.setAgentIdSilent("UnrealGoalEnvironmentControlServer");
		parameters.setWorldAddressSilent(LOCAL_HOST, CONTROL_SERVER_PORT);

		return parameters;
	}

	private void setWorldAddressSilent(String host, int port) {
		super.setWorldAddress(new SocketConnectionAddress(host, port));
	}

	private void setAgentIdSilent(String name) {
		super.setAgentId(new AgentId(name));
	}

}
