package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.executor.IAction;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;

/**
 * This class represents the arguments of the {@link PrimitiveCall}. {@link PrimitiveCall}
 * can reference either normal {@link IAction}, {@link ActionPattern} or {@link Competence}
 * that will get these arguments, mix them with their own {@link FormalParameters}
 * and create the values for the Action/C/AP.
 *
 * Arguments is stored as list of {@link Argument}, each argument is basically
 * (key; argument) pair. Arguments are sorted according to the key. What is the
 * key? Either a variable name or index of the argument in the list or
 * arguments. Variable names have '$' prefix, e.g. $name. The indexes are simply
 * string representations of the numbers, e.g. index 3 is represented as "3".
 *
 * Calling convention:
 * 
 * TODO: Properly document.
 * TODO: Properly test.
 * TODO: This probably doesn;t need to be abstract list
 *
 * Example: call run("street", speed=14.5) will create
 *
 * @author Honza Havlicek
 */
public class Arguments extends AbstractList<Arguments.Argument> {

    public static Arguments EMPTY = new Arguments();
    
    public static abstract class Argument {

        final private String parameterName;

        /**
         * Create paramter with name.
         *
         * @param parameterName
         */
        protected Argument(String parameterName) {
            this.parameterName = parameterName;
        }

        /**
         * Retrieve value of parameter.
         *
         * @return
         */
        public abstract Object getValue();

        /**
         * Is value of this parameter retrieved from variable or value?
         *
         * @return null if this is a value, string with name of parameter
         * variable else
         */
        public abstract String getParameterVariable();

        public String getParameterName() {
            return parameterName;
        }

        @Override
        public abstract String toString();

        @Override
        protected abstract Argument clone();
        
        /**
         * Create new argument object with specified name and value.
         *
         * @param argumentName name of argument.
         * @param value Object representation of the argument value
         * @return
         */
        public static Argument createValueArgument(String argumentName, Object value) {
            return new ValueArgument(argumentName, value);
        }
        
        /**
         * Create new argument with specified name and its value is defined by
         * the content of the variable.
         *
         * @param argumentName name of created argument
         * @param variableName name of variable that will be used to get the
         * value.
         * @return
         */
        public static Argument createVariableArgument(String argumentName, String variableName) {
            return new VariableArgument(argumentName, variableName);
        }
    }

    /**
     * This parameter is dependant on value of some kind of variable.
     */
    protected static final class VariableArgument extends Arguments.Argument {

        protected final String variableName;

        protected VariableArgument(int index, String variableName) {
            super(Integer.toString(index));
            this.variableName = variableName;
        }

        /**
         * New call parameter.
         *
         * @param parameterName name of parameter of called primitive
         * @param variableName variable where is stored the value passed to
         * primitive
         */
        protected VariableArgument(String parameterName, String variableName) {
            super(parameterName);
            this.variableName = variableName;
        }

        @Override
        public String getValue() {
            throw new UnsupportedOperationException("Not yet implemented");
        }

        @Override
        public String getParameterVariable() {
            return variableName;
        }

        @Override
        public String toString() {
            if (getParameterName().startsWith("$")) {
                return getParameterName() + "=" + variableName;
            }
            return variableName;
        }

        @Override
        protected VariableArgument clone() {
            return new VariableArgument(getParameterName(), variableName);
        }
    }

    /**
     * This parameter is static value, neverchanging.
     */
    protected static final class ValueArgument extends Arguments.Argument {

        private final Object value;

        /**
         * Create a value parameter that represents fixed value. <p> Sequence
         * number is used as name of this parameter. Since normal variables
         * starts with $, it won't mix.
         *
         * @param sequenceNumber number of this parameter in sequence of all
         * parameters. Starting from 0.
         * @param value value of parameter.
         */
        protected ValueArgument(int sequenceNumber, Object value) {
            super(Integer.toString(sequenceNumber));
            this.value = value;
        }

        protected ValueArgument(String parameterName, Object value) {
            super(parameterName);
            this.value = value;
        }

        @Override
        public Object getValue() {
            return value;
        }

        @Override
        public String getParameterVariable() {
            return null;
        }

        @Override
        public String toString() {
            if (getParameterName().startsWith("$")) {
                return getParameterName() + "=" + Result.toLap(value);
            }
            return Result.toLap(value);
        }

        @Override
        protected ValueArgument clone() {
            return new ValueArgument(getParameterName(), value);
        }
    }
    private List<Arguments.Argument> parameters = new ArrayList<Argument>();

    /**
     * Create a new list of call parameters. <p> Every added variable parameter
     * has to be checked against list of formal parameters.
     */
    public Arguments() {
    }

    /**
     * Copy constructor. Beware, this doesn't copy reference to the formal
     * parameters.
     *
     * @param parameters orginal
     */
    protected Arguments(Arguments parameters) {
        for (int i = 0; i < parameters.size(); ++i) {
            Argument parameter = parameters.get(i);
            Argument clone = parameter.clone();
            this.parameters.add(clone);
        }
    }

    @Override
    public synchronized Arguments.Argument get(int index) {
        return parameters.get(index);
    }

    @Override
    public synchronized int size() {
        return parameters.size();
    }

    @Override
    public void add(int index, Argument element) {
        parameters.add(index, element);
    }
    
    public synchronized boolean addFormal(Arguments.Argument element, FormalParameters formalParams) {
        // check if variable is defined in the context
        String parameterVariable = element.getParameterVariable();
        if (parameterVariable != null) {
            if (!formalParams.containsVariable(parameterVariable)) {
                throw new IllegalArgumentException("Variable \"" + parameterVariable + "\" is not defined in formal parameters (" + formalParams.toString() + ").");
            }
        }

        // next check that named parameter isn't duplicated
        for (int i = 0; i < parameters.size(); ++i) {
            String parameterName = parameters.get(i).getParameterName();
            // check if there isn't already variable name with same name
            if (parameterName != null && parameterName.equals(element.getParameterName())) {
                throw new IllegalArgumentException("Named parameter \"" + element.getParameterName() + "\" has already been defined.");
            }
        }
        return parameters.add(element);
    }

    /**
     * Get string representation of arguments (comma separated arguments,
     * compatible with lap).
     *
     * @return arguments separated with comma (e.g. |"Friends of peace.",
     * $when=now, $who=just_first_player|)
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (int argumentIndex = 0; argumentIndex < parameters.size(); ++argumentIndex) {
            if (argumentIndex != 0) {
                sb.append(',');
            }
            Argument argument = parameters.get(argumentIndex);
            sb.append(argument.toString());
        }
        return sb.toString();
    }
}
