package cz.cuni.amis.pogamut.sposh.engine;

import java.util.logging.Logger;

import cz.cuni.amis.pogamut.sposh.elements.PrimitiveCall;
import cz.cuni.amis.pogamut.sposh.engine.FireResult.Type;
import cz.cuni.amis.pogamut.sposh.executor.ActionResult;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;

/**
 * This is used to execute single primitive, like in DE or C.
 * @author Honza
 */
class ActionExecutor extends AbstractExecutor implements ElementExecutor {
    final private PrimitiveCall actionCall;
    
    final private FireResult.Type finishedResult;
    final private FireResult.Type runningResult;
    final private FireResult.Type runningOnceResult;
    final private FireResult.Type failedResult;
    
    final private Runnable finishedResultCallback;
    final private Runnable runningResultCallback;
    final private Runnable runningOnceResultCallback;
    final private Runnable failedResultCallback;

	

    /**
     * Create a primitive executor that will call a specified action
     * within context of parameters
     * 
     * @param actionCall actionCall to primitive
     * @param finishedResult what to return in case of {@link ActionResult#FINISHED}
     * @param execuringResult what to return in case of {@link ActionResult#RUNNING}
     * @param failedResult what to return in case of {@link ActionResult#FAILED}
     * @param ctx Context of this primitive. Should differ from parent, but isn't necessary
     * @param log logger to record actions of this executor, can be null
     */
    ActionExecutor(
            PrimitiveCall actionCall,
            FireResult.Type finishedResult,
            FireResult.Type runningResult,
            FireResult.Type runningOnceResult,
            FireResult.Type failedResult,
            VariableContext ctx,
            Logger log) {
        super(ctx, log);
        this.actionCall = actionCall;
        this.finishedResult = finishedResult;
        this.runningResult = runningResult;
        this.runningOnceResult = runningOnceResult;
        this.failedResult = failedResult;
        
        finishedResultCallback = null;
        runningResultCallback = null;
        runningOnceResultCallback = null;
        failedResultCallback = null;
    }
    
    /**
     * Create a primitive executor that will call a specified action
     * within context of parameters
     * 
     * @param actionCall actionCall to primitive
     * @param finishedResult what to return in case of {@link ActionResult#FINISHED}
     * @param execuringResult what to return in case of {@link ActionResult#RUNNING}
     * @param failedResult what to return in case of {@link ActionResult#FAILED}
     * @param finishedResultCallback called in case of {@link ActionResult#FINISHED}, may be null
     * @param runningResultCallback called in case of {@link ActionResult#RUNNING}, may be null
     * @param failedResultCallback called in case of {@link ActionResult#FAILED}, may be null
     * @param ctx Context of this primitive. Shoudl differ from parent, but isn't necessary
     * @param log logger to record actions of this executor, can be null
     */
    ActionExecutor(
            PrimitiveCall actionCall,
            FireResult.Type finishedResult,
            FireResult.Type runningResult,
            FireResult.Type runningOnceResult,
            FireResult.Type failedResult,
            Runnable finishedResultCallback,
            Runnable runningResultCallback,
            Runnable runningOnceResultCallback,
            Runnable failedResultCallback,
            VariableContext ctx,
            Logger log) {
        super(ctx, log);
        
        this.actionCall = actionCall;
        
        this.finishedResult = finishedResult;
        this.runningResult = runningResult;
        this.runningOnceResult = runningOnceResult;
        this.failedResult = failedResult;
        
        this.finishedResultCallback = finishedResultCallback;
        this.runningResultCallback = runningResultCallback;
        this.runningOnceResultCallback = runningOnceResultCallback;
        this.failedResultCallback = failedResultCallback;
    }


    /**
     * Fire the action and return FireResult(false), so don't continue
     * the execution.
     * @param workExecuter
     * @return failResult if result of primitive is empty or false, true otherwise
     */
    @Override
    public FireResult fire(IWorkExecutor workExecuter) {
        ActionResult result = workExecuter.executeAction(actionCall.getName(), ctx);
        switch(result) {
        case FAILED:
        	if (failedResultCallback != null) failedResultCallback.run();
        	return new FireResult(failedResult);
        	
        case FINISHED:
        	if (finishedResultCallback != null) finishedResultCallback.run();
        	return new FireResult(finishedResult);
        	
        case RUNNING:
        	if (runningResultCallback != null) runningResultCallback.run();
        	return new FireResult(runningResult);
        	
        case RUNNING_ONCE:
        	if (runningOnceResultCallback != null) runningOnceResultCallback.run();
        	return new FireResult(runningResult);
        	
        default:
        	throw new IllegalStateException("Unexpected ActionResult: " + result);
        }        
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + "[" + actionCall.getName() + "]";
    }

    @Override
    public TriggerResult getTriggerResult() {
        // XXX: Maybe null, I don;t have a consistent strategy for this
        return new TriggerResult(true);
    }
    
}
