package cz.cuni.amis.pogamut.sposh.elements;

import java.util.ArrayList;
import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;
import java.util.LinkedList;

/**
 * This is root of POSH plan in execution sense. In source of POSH plan, this is 
 * leaf of root. Every POSH plan can have only one DriveCollection.
 *
 * drive collection:  this is the root of the POSH hierarchy, and on every POSH
 * action-selection cycle, this reconsiders which goal the agent should be working on.
 * This is how a BOD agent can respond rapidly to changes in the environment.
 * 
 * @author Honza
 */
public final class DriveCollection extends PoshDummyElement {

    private String _name;
    private Goal _goal;
    private List<DrivePriorityElement> _elements;
    public static final String dcName = "dcName";

    /**
     * Create a new drive collection.
     * @param name Name of drive collection.
     * @param goal not null
     * @param drives not null
     */
    DriveCollection(String name, Goal goal, List<DrivePriorityElement> drives) {
        this._name = name;
        this._goal = goal;
        this._elements = drives;

        if (_goal != null) {
            this._goal.setParent(this);
        }
        for (DrivePriorityElement element : drives) {
            element.setParent(this);
        }
    }

    /**
     * Add new drive to the collection, assign parents and emit children.
     *
     * @param drive DrivePriorityElement to be added.
     */
    public void addDrivePriorityElement(DrivePriorityElement element) {
        element.setParent(this);
        _elements.add(element);

        emitChildNode(element);
    }

    /**
     * Set goal of the drive collection. In nearly all collection, the goal is fail.
     * If collection already has a goal, delete it, add new goal, assign parent and emit new children.
     *
     * @param goal
     */
    public void setGoalNode(Goal goal) {
        // Remove old goal
        if (this._goal != null)
            this._goal.remove();

        // Set new goal and set the parent of the goal to this
        this._goal = goal;
        goal.setParent(this);

        // Emit the goal and its children
        emitChildNode(goal);
    }

    public Goal getGoal() {
        return this._goal;
    }

    @Override
    public String toString() {
        String res;
        res = "\t(SDC " + this.getDriveCollectionName();
        res += " " + (_goal == null ? "" : _goal.toString());

        res += "\n\t\t(drives ";

        for (DrivePriorityElement drive : _elements) {

            res += "\n\t\t\t  (" + drive.toString() + ")";
        }
        res += "\n\t\t)";
        res += "\n\t)";
        return res;
    }

    @Override
    public List<PoshElement> getChildDataNodes() {
        List<PoshElement> children = new ArrayList<PoshElement>(_elements);
        
        if (_goal != null) {
            children.add(0, _goal);
        }

        return children;
    }

    public String getDriveCollectionName() {
        return _name;
    }

    /**
     * Set name of the drive collection.
     *
     * @param collectionName string that maches IDENT_PATTERN
     */
    public void setDriveCollectionName(String collectionName) {
        collectionName = collectionName.trim();
        if (collectionName.matches(IDENT_PATTERN)) {
            this._name = collectionName;
            firePropertyChange(dcName, null, collectionName);
        }
    }

    @Override
    public String getDisplayName() {
        return this.getDriveCollectionName();
    }

    @Override
    public boolean moveChild(PoshElement child, int relativePosition) {
        return moveNodeInList(this._elements, child, relativePosition);
    }
    
    public static final DataFlavor dataFlavor = new DataFlavor(DriveCollection.class, "drive-collection-node");

    @Override
    public DataFlavor getDataFlavor() {
        return dataFlavor;
    }

    @Override
    public void addChildDataNode(PoshElement newChild) {
        if (newChild instanceof Goal) {
            this.setGoalNode((Goal) newChild);
        } else if (newChild instanceof DrivePriorityElement) {
            this.addDrivePriorityElement((DrivePriorityElement) newChild);
        } else {
            throw new RuntimeException("Class " + newChild.getClass().getSimpleName() + " not accepted.");
        }
    }

    @Override
    public void neutralizeChild(PoshElement childNode) {
        if (childNode == this._goal) {
            this.setGoalNode(new Goal(new Sense("fail")));
        } else if (this._elements.contains(childNode)) {
            if (this._elements.size() <= 1) {
                this.addDrivePriorityElement(new DrivePriorityElement(new DriveElement("drive", new Triggers(new Sense("fail")), "do_nothing", null)));
            }
            this._elements.remove(childNode);
            childNode.remove();
        }
    }

    /**
     * Get unmodifiable list of all DrivePriorityElements.
     */
    public List<DrivePriorityElement> getPriorityElements() {
        return Collections.unmodifiableList(this._elements);
    }

    /**
     * Get list of all {@link DriveElement} in {@link DriveCollection} sorted
     * according to priority.
     * @return list of drives, sorted from highest priority to lowest.
     */
    public LinkedList<DriveElement> getDrives() {
        LinkedList<DriveElement> drives = new LinkedList<DriveElement>();

        for (DrivePriorityElement dpe : _elements) {
            for (DriveElement drive : dpe.getDriveElements()) {
                drives.add(drive);
            }
        }

        return drives;
    }


}
