package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.elements.CallParameters;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.Sense.Predicate;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.amis.pogamut.sposh.elements.Result;
import cz.cuni.amis.pogamut.sposh.elements.Sense.SenseCall;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import java.util.logging.Logger;

/**
 * Evaluate the primitive (either sesnee or action).
 * <p>
 * Possible format of the primitive (in the plan):
 * <ul>
 *  <li>primitive</li>
 *  <li>(primitive($variable,...))</li>
 *  <li>(primitive($variable,...) value)</li>
 *  <li>(primitive($variable,...) value predicate)</li>
 *  <li>(primitive)</li>
 *  <li>(primitive value)</li>
 *  <li>(primitive value predicate)</li>
 * </ul>
 * @author Honza
 */
final class SenseExecutor extends AbstractExecutor {

    private SenseCall senseCall;
    private Predicate predicate;
    /**
     * When evaluating the sense result, should I compare it with the operand?
     * XXX: if operand is true and predicate ==, don't compare
     */
    private final boolean compare = true;
    private Object operand;

    /**
     * Create new sense executor.
     * @param sense sense that is going to be executed
     * @param ctx variable context for the sense
     * @param log logger to record actions of this executor, can be null
     */
    SenseExecutor(Sense sense, VariableContext ctx, Logger log) {
        super(ctx, log);

        senseCall = sense.getSenseCall();
        predicate = sense.getPredicate();
        operand = sense.getOperand();
    }

    /**
     * Evaluate the primitive
     * @return true if it was evaluated successfully.
     */
    public boolean fire(IWorkExecutor workExecuter) {
        info("Fire: " + toString());

        Object primitiveResult = workExecuter.executePrimitive(senseCall.getName(), new VariableContext(ctx, senseCall.getParameters()));

        // If there is nothing to compare with, use only result.
        if (!compare) {
            return Result.isTrue(primitiveResult);
        }
        return evaluateComparison(primitiveResult, predicate, operand);
    }

    /**
     * Compare two values.
     * <p>
     * This thing deserves a lot of TLC. I try to abide to python convention:
     *
     * http://docs.python.org/reference/expressions.html#notin
     * http://docs.python.org/library/stdtypes.html
     *
     * <p>
     * numbers: numint, numfloat
     *
     * @param value1 nonempty value
     * @param predicate predicate that will be used for comparison
     * @param value2 nonempty value
     * @return
     */
    protected boolean evaluateComparison(Object operand1, Predicate predicate, Object operand2) {
        String comparison = operand1 + " " + predicate + " " + operand2;
        switch (predicate) {
            case DEFAULT:  // default predicate is equal
            case EQUAL:
                return Result.equal(operand1, operand2);
            case NOT_EQUAL:
                return !Result.equal(operand1, operand2);
            case LOWER:
                return Result.compare(operand1, operand2) < 0;
            case GREATER:
                return Result.compare(operand1, operand2) > 0;
            case LOWER_OR_EQUAL:
                return Result.equal(operand1, operand2) || Result.compare(operand1, operand2) < 0;
            case GREATER_OR_EQUAL:
                return Result.equal(operand1, operand2) || Result.compare(operand1, operand2) > 0;
            default:
                throw new IllegalArgumentException("Predicate operation \"" + predicate + "\" is implemented.");
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("SenseExecutor[(");

        sb.append(senseCall.getName());
        sb.append('(');
        boolean first = true;
        for (CallParameters.Parameter parameter : senseCall.getParameters()) {
            if (!first) {
                sb.append(", ");
            }
            sb.append(parameter.getParameterName());
            first = false;
        }
        sb.append(')');

        if (compare) {
            sb.append(' ');
            sb.append(predicate);

            sb.append(' ');
            sb.append(operand);
        }

        sb.append(')');
        sb.append(']');
        return sb.toString();
    }
}
