package cz.cuni.amis.pogamut.sposh.elements;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 * Test some examples of correct and incorrect posh plans.
 * @author Honza
 */
public class PoshParserTest {

    public PoshParserTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    private String getMethodName() {
        return Thread.currentThread().getStackTrace()[2].getMethodName();
    }

    private String loadPlan(String relativeResourcePath) throws IOException {
        String resourcePath = this.getClass().getPackage().getName().replace('.', '/') + '/' + relativeResourcePath;

        InputStream is = getClass().getClassLoader().getResourceAsStream(resourcePath);

        if (is == null) {
            fail("Unable to open resource \"" + resourcePath + "\"");
        }

        BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        StringBuilder sb = new StringBuilder();
        String line = null;

        while ((line = reader.readLine()) != null) {
            sb.append(line);
            sb.append('\n');
        }

        reader.close();

        return sb.toString();
    }

    private void testPoshPlan(String testName, String relativeResourcePath) throws IOException, ParseException {
        System.out.println("Test: " + testName);

        String plan = loadPlan(relativeResourcePath);
        PoshParser parser = new PoshParser(new StringReader(plan));
        parser.parsePlan();
    }

    @Test
    public void testAttackBot() throws Exception {
        testPoshPlan(getMethodName(), "testplans/joanna/attackbot.lap");
    }

    @Test
    public void testEducateMeMonk() throws Exception {
        testPoshPlan(getMethodName(), "testplans/joanna/educate-me+monk.lap");
    }

    @Test
    public void testPoshBotFollow() throws Exception {
        testPoshPlan(getMethodName(), "testplans/joanna/poshbotfollow.lap");
    }

    @Test
    public void testSheepDog() throws Exception {
        testPoshPlan(getMethodName(), "testplans/joanna/sheep-dog.lap");
    }

    @Test
    public void testStayGroom() throws Exception {
        testPoshPlan(getMethodName(), "testplans/joanna/stay-groom.lap");
    }

    @Test
    public void testCVarsPresnet() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/CompVarTest.lap");
    }

    @Test
    public void testCVarsNotPresnet() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/CompVarFail.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("<VARIABLE>")) {
                return;
            }
            return;
        }
        fail("Should throw exception, no parameters");
    }

    @Test
    public void testCVarsMissingDefault() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/CompVarMissingDefault.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("\"=\"")) {
                return;
            }
            return;
        }
        fail("Should throw exception, no default value for parameter");
    }

    @Test
    public void testAPVarsMissingParameters() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/APVarsMissingParameters.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("\"murder\"")) {
                return;
            }
            return;
        }
        fail("Should throw exception, no default value for parameter");
    }

    @Test
    public void testAPVarsParameters() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/APVarsParameters.lap");
    }

    @Test
    public void testSenseVarsInComp() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/SenseVarsInComp.lap");
    }

    @Test
    public void testSensesVarsInCompContextFail() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/SensesVarsInCompContextFail.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("$killbot")) {
                return;
            }
        }
        fail("Should throw exception, variable $killbot in sense call not defined");
    }

    @Test
    public void testAPActionVar() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/APActionVars.lap");
    }

    @Test
    public void testAPActionVarsContextFail() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/APActionVarsContextFail.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("$where")) {
                return;
            }
        }
        fail("Should throw exception, variable $where not defined in AP context.");
    }

    @Test
    public void testCompDuplicateVars() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/CompDuplicateVars.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("$duplicateVariable")) {
                return;
            }
        }
        fail("Should throw exception, competence has declared two $duplicateVariable variables.");
    }

    // call comptence with variables, basically same as calling actions
    @Test
    public void testCallC() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/CallC.lap");
    }

    // call comptence with variables, basically same as calling actions
    @Test
    public void testCallPrimitiveFromC() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/CallPrimitiveFromC.lap");
    }

    // call comptence with variables, basically same as calling actions
    @Test
    public void testCallCFromDC() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/CallCFromDC.lap");
    }

    @Test
    public void testCallPrimitiveFromDC() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/CallPrimitiveFromDC.lap");
    }

    @Test
    public void testSenseTriggerVars() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/SenseTriggerVars.lap");
    }

    @Test
    public void testMissingVars() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/MissingVars.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("$var1") &
                    ex.getMessage().contains("\"elements\"") &
                    ex.getMessage().contains("\"goal\"") &
                    ex.getMessage().contains("\"hours\"") &
                    ex.getMessage().contains("\"minutes\"") &
                    ex.getMessage().contains("\"seconds\"")) {
                return;
            }
        }
        fail("Should throw exception, missing \"vars\" keyword.");
    }

    @Test
    public void testActionNamedVariable() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/ActionNamedVariable.lap");
    }

    @Test
    public void testNamedActionsFail() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/NamedActionsFail.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("unnamed") &
                    ex.getMessage().contains("parameter") &
                    ex.getMessage().contains("named")) {
                return;
            }
        }
        fail("Should throw exception, unnamed parameter after named parameter.");
    }

    @Test
    public void testNamedParameterDuplicate() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/NamedParameterDuplicate.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("\"$variable\"") &
                    ex.getMessage().contains("already") &
                    ex.getMessage().contains("defined")) {
                return;
            }
        }
        fail("Should throw exception, two named parameters were defined.");
    }

    @Test
    public void testNamedCallTest() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/NamedCallTest.lap");
    }

    @Test
    public void testTestStringVariable() throws Exception {
        testPoshPlan(getMethodName(), "testplans/vars/TestStringVariable.lap");
    }

    @Test
    public void testVarStringFail() throws Exception {
        try {
            testPoshPlan(getMethodName(), "testplans/vars/VarStringFail.lap");
        } catch (ParseException ex) {
            if (ex.getMessage().contains("Encountered") &&
                    ex.getMessage().contains("\"\\\"\"")) {
                return;
            }
        }
    }
}
