package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.engine.timer.SystemClockTimer;
import cz.cuni.amis.pogamut.sposh.engine.timer.ITimer;
import cz.cuni.amis.pogamut.sposh.elements.DriveCollection;
import cz.cuni.amis.pogamut.sposh.elements.DriveElement;
import cz.cuni.amis.pogamut.sposh.elements.DrivePriorityElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.amis.pogamut.sposh.engine.PoshEngine.EvaluationResult;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

/**
 * This class is responsible for valuating the DriveCollection element.
 * @author Honza
 */
final class DCExecutor extends AbstractExecutor {
    private SenseListExecutor goalExecutor;
    private List<DEExecutor> deExecutors = new ArrayList<DEExecutor>();
    private ITimer timer;


    /**
     * Create DCExecutor with SystemClockTimer.
     * @param dc
     * @param ctx
     */
    protected DCExecutor(PoshPlan plan, VariableContext ctx) {
        this(plan, ctx, new SystemClockTimer(), null);
    }

    protected DCExecutor(PoshPlan plan, VariableContext ctx, ITimer timer, Logger log) {
        super(ctx, log);

        DriveCollection dc = plan.getDriveCollection();
        this.timer = timer;
        this.goalExecutor = new SenseListExecutor(dc.getGoal(), ctx, log);

        for (DrivePriorityElement dpe : dc.getPriorityElements()) {
            for (DriveElement de : dpe.getDriveElements()) {
                deExecutors.add(new DEExecutor(plan, de, ctx, log));
            }
        }
    }

    
    public synchronized EvaluationResult fire(IWorkExecutor workExecuter) {
        if (goalExecutor.fire(workExecuter, false)) {
            return EvaluationResult.GOAL_SATISFIED;
        }
        for (DEExecutor deExecutor : deExecutors) {
            if (deExecutor.isReady(timer.getTime(), workExecuter)) {
                deExecutor.fire(workExecuter, timer);
                return EvaluationResult.ELEMENT_FIRED;
            }
        }
        return EvaluationResult.NO_ELEMENT_FIRED;
    }

    ElementStackTrace getStackForDE(int index) {
        return deExecutors.get(index).getStackTrace();
    }

    /**
     * Get ElementStackTrace for the drive element with name "name."
     * If there are two drive elements with same name, throw IllegalStateException.
     *
     * @param name name of drive element we are looking for
     * @return stacktrace for element if element with such name exists, else null.
     */
    ElementStackTrace getStackForDE(String name) {
        DEExecutor result = null;
        for (DEExecutor de : deExecutors) {
            boolean equal = de.getName() == null ? name == null : de.getName().equals(name);
            if (equal) {
                if (result == null) {
                    result = de;
                } else {
                    throw new IllegalStateException("Two drive elements with name \"" + name + "\".");
                }
            }
        }
        if (result != null)
            return result.getStackTrace();

        return null;
    }

}
