package cz.cuni.pogamut.shed.presenter;

import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Competence;
import cz.cuni.amis.pogamut.sposh.elements.CompetenceElement;
import cz.cuni.amis.pogamut.sposh.elements.DriveElement;
import cz.cuni.amis.pogamut.sposh.elements.FormalParameters;
import cz.cuni.amis.pogamut.sposh.elements.ParseException;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshParser;
import cz.cuni.amis.pogamut.sposh.elements.PrimitiveCall;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.amis.pogamut.sposh.exceptions.CycleException;
import cz.cuni.amis.pogamut.sposh.exceptions.DuplicateNameException;
import cz.cuni.amis.pogamut.sposh.exceptions.InvalidNameException;
import cz.cuni.pogamut.shed.widget.editor.APEditorProvider;
import cz.cuni.pogamut.shed.widget.editor.CEditorProvider;
import java.io.StringReader;
import java.text.MessageFormat;
import org.netbeans.api.visual.action.InplaceEditorProvider;
import org.netbeans.api.visual.action.TextFieldInplaceEditor;
import org.netbeans.api.visual.widget.Widget;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;

/**
 * Factroy for creating various {@link TextFieldInplaceEditor inplace editors}.
 * To use these editors in the widget, you must add them to the action chain of
 * a widget using {@link ActionFactory#createInplaceEditorAction(org.netbeans.api.visual.action.TextFieldInplaceEditor)
 * }.
 *
 * @author Honza
 */
public class ShedInplaceEditorFactory {

    /**
     * Create editor that will edit the @sense, its name, predicate and compared
     * value.
     *
     * @param sense {@link Sense} that will have its name changed.
     * @return
     */
    public static TextFieldInplaceEditor createSenseEditor(Sense sense) {
        return new SenseInplaceEditor(sense);
    }

    /**
     * Create editor that will edit the name of the @drive.
     *
     * @param drive Drive that will have its name changed.
     * @return
     */
    public static TextFieldInplaceEditor createDriveEditor(DriveElement drive) {
        return new DriveInplaceEditor(drive);
    }

    /**
     * Create editor that will edit the name of the @action.
     *
     * @param action Action that will have its name changed.
     * @return
     */
    public static TextFieldInplaceEditor createActionEditor(TriggeredAction action) {
        return new ActionInplaceEditor(action);
    }

    /**
     * Create editor that will edit the name of the @choice.
     *
     * @param choice {@link CompetenceElement} that will have its name changed.
     * @return
     */
    public static TextFieldInplaceEditor createChoiceEditor(CompetenceElement choice) {
        return new ChoiceInplaceEditor(choice);
    }

    /**
     * Create editor for action referencing action pattern. The editor can
     * rename the AP, add/remove parameters to the AP and add/remove arguments
     * passed by the action to the AP.
     *
     * @param actionPattern Action pattern that is referenced by the action.
     * @param referencingAction action that is referencing the AP.
     * @return Created editor provider.
     */
    public static InplaceEditorProvider createActionPatternEditor(ActionPattern actionPattern, TriggeredAction referencingAction) {
        assert actionPattern.getName().equals(referencingAction.getName());
        return new APEditorProvider(actionPattern, referencingAction);
    }

    /**
     * Create inplace editor for action referencing competence. The editor can
     * rename the competence, add/remove parameters of the competence and
     * add/remove arguments passed by the action to the competence.
     *
     * @param competence Competence that is referenced by the action.
     * @param referencingAction action that is referencing the competence.
     * @return Created editor provider.
     */
    public static InplaceEditorProvider createCompetenceEditor(Competence competence, TriggeredAction referencingAction) {
        assert competence.getName().equals(referencingAction.getName());
        return new CEditorProvider(competence, referencingAction);
    }

    private static class ActionInplaceEditor implements TextFieldInplaceEditor {

        private final TriggeredAction action;

        ActionInplaceEditor(TriggeredAction action) {
            this.action = action;
        }

        @Override
        public boolean isEnabled(Widget widget) {
            return true;
        }

        @Override
        public String getText(Widget widget) {
            return action.toString();
        }

        private void notify(String message) {
            NotifyDescriptor.Message infoMessage = new NotifyDescriptor.Message(message, NotifyDescriptor.INFORMATION_MESSAGE);
            DialogDisplayer.getDefault().notify(infoMessage);
        }

        /**
         * Get available variables that can be used as arguments during the
         * actioncall, e.g all action calls in (AP nya vars($a=1, $b="4") (
         * meow($b) scratch) ) can use variables $a and $b in its actioncall,
         * just like the meow did, in C it is similar, you can use declared
         * variables of C in the senses and actions.
         *
         * @return List of available formal parameters of the parent element of
         * the action. If there is no parent, return empty.
         */
        private FormalParameters getParameters() {
            PoshElement parent = action.getParent();
            assert parent != null : "I should never use variables in node that is not part of element";

            if (parent instanceof ActionPattern) {
                ActionPattern ap = (ActionPattern) parent;
                return ap.getParameters();
            }
            if (parent instanceof DriveElement) {
                return new FormalParameters();
            }
            if (parent instanceof CompetenceElement) {
                CompetenceElement cel = (CompetenceElement) parent;
                Competence competence = cel.getParent();
                return competence.getParameters();
            }
            throw new IllegalStateException("Unexpected parent: " + parent.getClass().getCanonicalName());
        }

        private PrimitiveCall parseActionCall(String text) throws ParseException {
            PoshParser parser = new PoshParser(new StringReader(text));
            return parser.senseCall(getParameters());
        }

        @Override
        public void setText(Widget widget, String input) {
            try {
                PrimitiveCall call = parseActionCall(input);
                action.setActionName(call.getName());
                action.setArguments(call.getParameters());
            } catch (InvalidNameException ex) {
                String errorMessage = MessageFormat.format("Action name \"{0}\" is not valid", input);
                notify(errorMessage);
            } catch (CycleException ex) {
                String errorMessage = MessageFormat.format("Action referencing to \"{0}\" would cause a cycle.", input);
                notify(errorMessage);
            } catch (ParseException ex) {
                String errorMessage = MessageFormat.format("Unable to parse input \"{0}\".\nError message: {1}", input, ex.getMessage());
                notify(errorMessage);
            }
        }
    };

    private static class SenseInplaceEditor implements TextFieldInplaceEditor {

        private final Sense sense;

        private SenseInplaceEditor(Sense sense) {
            this.sense = sense;
        }

        @Override
        public boolean isEnabled(Widget widget) {
            return true;
        }

        @Override
        public String getText(Widget widget) {
            String senseCall = sense.getSenseCall().toString();
            if (sense.getOperand() == Boolean.TRUE && sense.getPredicate() == Sense.Predicate.DEFAULT) {
                return senseCall;
            }
            String value = sense.getValueString();
            String predicate = sense.getPredicate().toString();
            return senseCall + ' ' + value + ' ' + predicate;
        }

        private void notify(String message) {
            NotifyDescriptor.Message infoMessage = new NotifyDescriptor.Message(message, NotifyDescriptor.INFORMATION_MESSAGE);
            DialogDisplayer.getDefault().notify(infoMessage);
        }

        @Override
        public void setText(Widget widget, String newSense) {
            try {
                sense.parseSense(newSense);
            } catch (ParseException ex) {
                String errorMessage = MessageFormat.format("Unable to parse the sense \"{0}\".\nError message: {1}.", newSense, ex.getMessage());
                notify(errorMessage);
            }
        }
    }

    private static class DriveInplaceEditor implements TextFieldInplaceEditor {

        private final DriveElement drive;

        public DriveInplaceEditor(DriveElement drive) {
            this.drive = drive;
        }

        @Override
        public boolean isEnabled(Widget widget) {
            return true;
        }

        @Override
        public String getText(Widget widget) {
            return drive.getName();
        }

        private void notify(String message) {
            NotifyDescriptor.Message infoMessage = new NotifyDescriptor.Message(message, NotifyDescriptor.INFORMATION_MESSAGE);
            DialogDisplayer.getDefault().notify(infoMessage);
        }

        @Override
        public void setText(Widget widget, String newDriveName) {
            try {
                drive.setDriveName(newDriveName);
            } catch (DuplicateNameException ex) {
                String message = MessageFormat.format("Drive with name \"{0}\" is already present.", newDriveName);
                notify(message);
            } catch (InvalidNameException ex) {
                String message = MessageFormat.format("Drive name \"{0}\" is not valid.", newDriveName);
                notify(message);
            }
        }
    }

    private static class ChoiceInplaceEditor implements TextFieldInplaceEditor {

        private final CompetenceElement choice;

        public ChoiceInplaceEditor(CompetenceElement choice) {
            this.choice = choice;
        }

        @Override
        public boolean isEnabled(Widget widget) {
            return true;
        }

        @Override
        public String getText(Widget widget) {
            return choice.getName();
        }

        private void notify(String message) {
            NotifyDescriptor.Message infoMessage = new NotifyDescriptor.Message(message, NotifyDescriptor.INFORMATION_MESSAGE);
            DialogDisplayer.getDefault().notify(infoMessage);
        }

        @Override
        public void setText(Widget widget, String newChoiceName) {
            try {
                choice.setName(newChoiceName);
            } catch (DuplicateNameException ex) {
                String message = MessageFormat.format("Choice name \"{0}\" already exists.", newChoiceName);
                notify(message);
            } catch (InvalidNameException ex) {
                String message = MessageFormat.format("Choice name \"{0}\" is not valid.", newChoiceName);
                notify(message);
            }
        }
    }
}
