package cz.cuni.pogamut.shed.view;

import cz.cuni.amis.pogamut.sposh.elements.*;
import cz.cuni.amis.pogamut.sposh.exceptions.CycleException;
import cz.cuni.amis.pogamut.sposh.executor.IAction;
import cz.cuni.amis.pogamut.sposh.executor.ISense;
import cz.cuni.pogamut.posh.PoshDataObject;
import cz.cuni.pogamut.posh.PoshEditorSupport;
import cz.cuni.pogamut.posh.explorer.Crawler;
import cz.cuni.pogamut.posh.explorer.CrawlerExplorerFactory;
import cz.cuni.pogamut.posh.explorer.CrawlerFactory;
import cz.cuni.pogamut.posh.explorer.CrawlerListener;
import cz.cuni.pogamut.posh.explorer.Explorer;
import cz.cuni.pogamut.posh.explorer.NameMapCrawler;
import cz.cuni.pogamut.posh.explorer.PrimitiveData;
import cz.cuni.pogamut.shed.presenter.ShedPresenter;
import cz.cuni.pogamut.shed.widget.LapSceneFactory;
import cz.cuni.pogamut.shed.widget.ShedScene;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import javax.swing.*;
import javax.swing.text.BadLocationException;
import javax.swing.text.StyledDocument;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.core.spi.multiview.CloseOperationState;
import org.netbeans.core.spi.multiview.MultiViewElement;
import org.netbeans.core.spi.multiview.MultiViewElementCallback;
import org.netbeans.core.spi.multiview.MultiViewFactory;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.awt.UndoRedo;
import org.openide.loaders.DataObject;
import org.openide.util.Exceptions;
import org.openide.util.Lookup;

/**
 * MV component showing the lap plan as tree. Basically visual mode of modifying
 * the plan.
 *
 * @author HonzaH
 */
public final class LapTreeMVElement extends JSplitPane implements MultiViewElement {

    /**
     * Toolbar of this MVE.
     */
    private final JToolBar toolbar;
    /**
     * MV callback.
     */
    private MultiViewElementCallback callback;
    /**
     * Data object that will be edited in this element.
     */
    private final PoshDataObject dObj;
    /**
     * Tree form of lap plan.
     */
    private final PoshPlan lapTree;
    /**
     * Sceen that will be showing the lap tree as widget tree.
     */
    private final ShedScene scene;
    /**
     * Default name of the drive collection that is displayed
     */
    private static final String DEFAULT_DC_NAME = "life";
    /**
     * Global listener that registers all elements of the tree and if anything
     * changes, it marks data object as modified.
     */
    private final TreeModificationListener saveListener;
    /**
     * Scroll pane that contains the {@link ShedScene#createView() view of the scene},
     * the left side of this element.
     */
    private final JScrollPane sceneScrollPane;
    /**
     * Tabbed panel with explorers for APs, Cs, actions and senses({@link CrawlerExplorerFactory}).
     * The right side of the element.
     */
    private final JTabbedPane explorer;
    /**
     * Listener on crawled {@link IAction}s in the classpath of the project.
     * When it gets crawled data from the explorer, it sends it to the {@link ShedPresenter},
     * which redistributes it to interested parties.
     */
    private final CrawlerListener<PrimitiveData> actionNameMapListener;
    /**
     * Listener on crawled {@link ISense}s in the classpath of the project. When
     * it gets crawled data from the explorer, it sends it to the {@link ShedPresenter},
     * which redistributes it to interested parties.
     */
    private final CrawlerListener<PrimitiveData> sensesNameMapListener;

    private final Explorer<Competence> competencesExplorer;
    private final Explorer<ActionPattern> actionPatternsExplorer;
    
    /**
     * Create new view for lap tree. This one only creates defualtlap tree
     * element for passed data object.
     *
     * @param dObj object that will be edited using this element.
     */
    LapTreeMVElement(PoshDataObject dObj) {
        super(JSplitPane.HORIZONTAL_SPLIT);

        this.dObj = dObj;
        this.lapTree = LapElementsFactory.createPlan(DEFAULT_DC_NAME);
        this.saveListener = new TreeModificationListener(lapTree, dObj.getEditorSupport().getDocument());

        this.scene = LapSceneFactory.createShedScene(this.lapTree);

        this.toolbar = createToolbarRepresentation();

        this.sceneScrollPane = new JScrollPane();
        this.sceneScrollPane.setViewportView(scene.createView());

        this.setLeftComponent(this.sceneScrollPane);

        this.explorer = new JTabbedPane();
        this.explorer.setPreferredSize(new Dimension(300, 1024));
        
        this.competencesExplorer = CrawlerExplorerFactory.createCompetenceExplorer(lapTree);
        explorer.addTab("Competences", competencesExplorer);

        this.actionPatternsExplorer = CrawlerExplorerFactory.createAPExplorer(lapTree);
        explorer.addTab("Action patterns", actionPatternsExplorer);

        Project project = getProject();
        actionNameMapListener = new NameMapCrawler(scene.getPresenter());
        sensesNameMapListener = new NameMapCrawler(scene.getPresenter());
        if (project != null) {
            explorer.addTab("Actions", CrawlerExplorerFactory.createActionsExplorer(project, actionNameMapListener));
            explorer.addTab("Senses", CrawlerExplorerFactory.createSensesExplorer(project, sensesNameMapListener));
        }
        this.setRightComponent(explorer);
        this.setResizeWeight(1.0);
    }

    private Project getProject() {
        return FileOwnerQuery.getOwner(dObj.getPrimaryFile());
    }

    @Override
    public JComponent getVisualRepresentation() {
        // called every time the view is activated
        return this;
    }

    @Override
    public JComponent getToolbarRepresentation() {
        return toolbar;
    }

    @Override
    public Action[] getActions() {
        if (callback != null) {
            // default actions of TC
            return callback.createDefaultActions();
        }
        return new Action[0];
    }

    @Override
    public Lookup getLookup() {
        return dObj.getNodeDelegate().getLookup();
    }

    /**
     * Called only when enclosing multi view top component was closed before and
     * now is opened again for the first time.
     */
    @Override
    public void componentOpened() {
        //  Subclasses will usually perform initializing tasks here. 
        this.saveListener.register();
    }

    /**
     * Called when this MultiViewElement is about to be shown. That can happen
     * when switching the current perspective/view or when the topcomonent
     * itself is shown for the first time.
     */
    @Override
    public void componentShowing() {
        PoshPlan documentLapTree;
        try {
            documentLapTree = dObj.parseLapPlan();
        } catch (ParseException ex) {
            sceneScrollPane.setViewportView(new JTextField(ex.getMessage()));
            return;
        }

        sceneScrollPane.setViewportView(scene.getView());
        
        lapTree.synchronize(documentLapTree);
        scene.update();
        
        // update competences and AP
        competencesExplorer.recrawlElements();
        actionPatternsExplorer.recrawlElements();
    }

    /**
     * Called only when multi view top component was closed.
     */
    @Override
    public void componentClosed() {
        // Do the cleanup
        this.saveListener.unregister();
    }

    @Override
    public void componentHidden() {
    }

    @Override
    public void componentActivated() {
    }

    @Override
    public void componentDeactivated() {
    }

    @Override
    public UndoRedo getUndoRedo() {
        return UndoRedo.NONE;
    }

    @Override
    public void setMultiViewCallback(MultiViewElementCallback mvec) {
        this.callback = mvec;
    }

    @Override
    public CloseOperationState canCloseElement() {
        // MVTC will be closed when all MVE return OK, so we let text editor handle closing.
//        return CloseOperationState.STATE_OK;
        if (dObj.isModified()) {
            AbstractAction saveAction = new AbstractAction() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    try {
                        dObj.getEditorSupport().saveDocument();
                    } catch (IOException ex) {
                        Exceptions.printStackTrace(ex);
                    }
                }
            };
            AbstractAction discardAction = new AbstractAction() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    dObj.getEditorSupport().onCloseDiscard();
                }
            };
            String messageSave = dObj.getEditorSupport().messageSave();

            return MultiViewFactory.createUnsafeCloseState(messageSave, saveAction, discardAction);
        }
        return CloseOperationState.STATE_OK;

    }

    private JToolBar createToolbarRepresentation() {
        JToolBar tb = new JToolBar();
        tb.add(new AbstractAction("Create sense") {

            @Override
            public void actionPerformed(ActionEvent e) {
                NotifyDescriptor.InputLine desc = new NotifyDescriptor.InputLine("Get name of sense", "Add new sense to goal od DC");
                DialogDisplayer.getDefault().notify(desc);
                if (desc.getValue() != NotifyDescriptor.OK_OPTION) {
                    return;
                }
                lapTree.getDriveCollection().getGoal().add(LapElementsFactory.createSense(desc.getInputText()));
            }
        });
        return tb;
    }
}

class TreeModificationListener implements PoshElementListener {

    private final PoshPlan lapTree;
    private final StyledDocument document;
    private int balance = 0;

    public TreeModificationListener(PoshPlan lapTree, StyledDocument document) {
        this.lapTree = lapTree;
        this.document = document;
    }

    public void register() {
        registerBranch(lapTree, true);
    }

    public void unregister() {
        registerBranch(lapTree, false);
        if (balance != 0) {
            String message = MessageFormat.format("Balance of register/unregister pairs for global listener is {0}, should be 0. If you can reproduce, report.", balance);
            NotifyDescriptor.Message desc = new NotifyDescriptor.Message(message);
            DialogDisplayer.getDefault().notify(desc);
        }
    }

    private void registerElement(PoshElement element, boolean register) {
        if (register) {
            element.addElementListener(this);
            ++balance;
        } else {
            element.removeElementListener(this);
            --balance;
        }
    }

    private void registerBranch(PoshElement<?, ?> branchRoot, boolean register) {
        registerElement(branchRoot, register);
        for (PoshElement child : branchRoot.getChildDataNodes()) {
            registerBranch(child, register);
        }
    }

    @Override
    public void childElementAdded(PoshElement parent, PoshElement child) {
        registerBranch(child, true);
        notifyTreeModified();
    }

    @Override
    public void childElementMoved(PoshElement parent, PoshElement child, int oldIndex, int newIndex) {
        notifyTreeModified();
    }

    @Override
    public void childElementRemoved(PoshElement parent, PoshElement child, int removedChildPosition) {
        registerBranch(child, false);
        notifyTreeModified();
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        notifyTreeModified();
    }

    private void notifyTreeModified() {
        try {
            document.remove(0, document.getLength());
            document.insertString(0, lapTree.toString(), null);
        } catch (BadLocationException ex) {
            Exceptions.printStackTrace(ex);
        }
    }
}