/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package cz.cuni.amis.pogamut.ut2004.ut2004testfw;

import cz.cuni.amis.pogamut.ut2004.ut2004testfw.config.BotTemplate;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.config.MatchConfig;
import cz.cuni.amis.pogamut.ut2004.tournament.match.UT2004Match;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.measure.IMeasure;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.utils.CsvReader;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.utils.CsvReader.CsvRow;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * A class for executing sets of matches.
 * @author tommassino
 */
public class MatchesExecutor {

    /**
     * Reading/writing buffer size.
     */
    private static final int BUFFER_SIZE = 1024;
    /**
     * The match sets to run.
     */
    private ArrayList<MatchConfig> matchSets;
    /**
     * The measures to analyze.
     */
    private ArrayList<IMeasure> measures;
    /**
     * A map with bots mapped to their id's.
     */
    private HashMap<String, BotTemplate> botMap;
    /**
     * Path to UT2004.
     */
    private String utPath;
    /**
     * The output directory.
     */
    private String outputDirectory;
    /**
     * The current test id.
     */
    private int testId;

    /**
     * Prepares the MatchesExecutor for match execution.
     * @param matchSets The match sets to run.
     * @param bots The bots that are to be used in the matches.
     * @param measures The measures to analyze.
     * @param utPath Path to UT2004.
     * @param outputDirectory The output directory.
     */
    public MatchesExecutor(ArrayList<MatchConfig> matchSets, ArrayList<BotTemplate> bots, ArrayList<IMeasure> measures, String utPath, String outputDirectory) {
        this.matchSets = matchSets;
        this.measures = measures;
        this.outputDirectory = outputDirectory;
        this.utPath = utPath;
        testId = generateTestId();
        botMap = new HashMap<String, BotTemplate>();
        for (BotTemplate bot : bots) {
            if (botMap.containsKey(bot.getId())) {
                System.err.println("Duplicate bot id: " + bot.getId());
            } else {
                botMap.put(bot.getId(), bot);
            }
        }
    }

    /**
     * A method that analyzes a match by parsing the results with the measures.
     * @param run The run id to analyze.
     * @param match The match to analyze.
     * @param config Config of the match.
     * @param outputPath Path to the results.
     */
    private void analyzeMatch(int run, UT2004Match match, MatchConfig config, String outputPath) {
        for (BotTemplate bot : botMap.values()) {
            if (!bot.observe()) {
                continue;
            }

            //find capture files
            File[] candidates = match.getOutputPath("bots").listFiles();
            ArrayList<ArrayList<File>> reports = new ArrayList<ArrayList<File>>();
            for (File can : candidates) {
                if (can.getName().matches(bot.getId() + "[0-9]*_[0-9]{3}\\.csv")) {
                    String sid = can.getName();
                    sid = sid.substring(sid.lastIndexOf("_") + 1, sid.indexOf(".csv"));
                    int iid = Integer.parseInt(sid);
                    while (reports.size() <= iid) {
                        reports.add(new ArrayList<File>());
                    }
                    reports.get(iid).add(can);
                }
            }
            if (reports.isEmpty()) {
                return;
            }
            Collections.sort(reports, new Comparator<ArrayList<File>>() {

                @Override
                public int compare(ArrayList<File> t, ArrayList<File> t1) {
                    if (t == null) {
                        return 1;
                    } else if (t1 == null) {
                        return -1;
                    }
                    return (int) (t1.get(0).lastModified() - t.get(0).lastModified());
                }
            });
            ArrayList<File> reportList = reports.get(0);

            for (IMeasure measure : measures) {
                System.out.println(measure.getName());
                measure.init();
                try {
                    for (File report : reportList) {
                        Logger.getLogger(PerformanceReport.class.getName()).log(Level.INFO, "Reading report {0}", report.getName());
                        CsvReader reader = new CsvReader(report.getAbsolutePath(), ";");
                        CsvRow row;
                        while ((row = reader.readRow()) != null) {
                            measure.next(row);
                        }
                        measure.reportEnd();
                    }
                } catch (FileNotFoundException ex) {
                    Logger.getLogger(PerformanceReport.class.getName()).log(Level.SEVERE, null, ex);
                } catch (IOException ex) {
                    Logger.getLogger(PerformanceReport.class.getName()).log(Level.SEVERE, null, ex);
                }
                try {
                    BufferedWriter out = new BufferedWriter(new FileWriter(outputPath, true));
                    out.write(testId + ";" + measure.getName() + ";" + config.getId() + ";" + run + ";" + measure.getMeasure() + "\n");
                    out.close();
                } catch (IOException ex) {
                    Logger.getLogger(PerformanceReport.class.getName()).log(Level.SEVERE, null, ex);
                }
            }

            File od = match.getOutputPath();
            File replay = null;
            for (File f : od.listFiles()) {
                if (f.getName().endsWith(".demo4")) {
                    replay = f;
                    break;
                }
            }
            if (replay != null) {
                System.out.println(replay.getAbsolutePath());
                File out = new File(outputDirectory, config.getId() + "-" + run + "-replay.demo4");
                System.out.println(out.getAbsolutePath());
                FileInputStream fis = null;
                FileOutputStream fos = null;
                try {
                    fis = new FileInputStream(replay);
                    fos = new FileOutputStream(out);
                    byte[] buf = new byte[BUFFER_SIZE];
                    int i = 0;
                    while ((i = fis.read(buf)) != -1) {
                        fos.write(buf, 0, i);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                } finally {
                    try {
                        if (fis != null) {
                            fis.close();
                        }
                        if (fos != null) {
                            fos.close();
                        }
                    } catch (IOException ex) {
                    }
                }
            }
        }
    }

    /**
     * Executes all matches and analyzes them.
     */
    public void executeMatches() {
        Logger.getLogger(MatchesExecutor.class.getName()).log(Level.INFO, "Starting matches");
        for (MatchConfig config : matchSets) {
            for (int i = 0; i < config.getRuns(); i++) {
                final UT2004Match match = config.createMatch(utPath, botMap);
                match.getLog().setLevel(Level.SEVERE);
                match.getLog().addConsoleHandler();
                match.cleanUp();

                try {
                    match.run();
                } catch (Exception e) {
                    e.printStackTrace();
                }
                analyzeMatch(i, match, config, outputDirectory + File.separator + "results");
                File[] output = match.getOutputPath("bots").listFiles();
                for (File res : output) {
                    res.delete();
                }

            }
        }
    }

    /**
     * Generates an id of the test, if there was no test before, the id is 1, otherwise it increments the last test id.
     * @return the test id
     */
    private int generateTestId() {
        File idFile = new File(outputDirectory + File.separator + "id");
        int id = -1;
        if (!idFile.exists()) {
            id = 1;
        } else {
            BufferedReader reader = null;
            try {
                reader = new BufferedReader(new FileReader(idFile));
                id = Integer.parseInt(reader.readLine()) + 1;
            } catch (FileNotFoundException ex) {
                id = 1;
                Logger.getLogger(MatchesExecutor.class.getName()).log(Level.SEVERE, null, ex);
            } catch (IOException ex) {
                id = 1;
                Logger.getLogger(MatchesExecutor.class.getName()).log(Level.SEVERE, null, ex);
            } finally {
                if (reader != null) {
                    try {
                        reader.close();
                    } catch (IOException ex) {
                    }
                }
            }
        }
        BufferedWriter out = null;
        try {
            out = new BufferedWriter(new FileWriter(idFile, false));
            out.write(id + "\n");
        } catch (IOException ex) {
            Logger.getLogger(MatchesExecutor.class.getName()).log(Level.SEVERE, null, ex);
        } finally {
            if (out != null) {
                try {
                    out.close();
                } catch (IOException ex) {
                }
            }
        }
        File resultFile = new File(outputDirectory + File.separator + "results");
        if (!resultFile.exists()) {
            out = null;
            try {
                out = new BufferedWriter(new FileWriter(resultFile, false));
                out.write("testId;measureName;matchId;runId;measureResult\n");
            } catch (IOException ex) {
                Logger.getLogger(MatchesExecutor.class.getName()).log(Level.SEVERE, null, ex);
            } finally {
                if (out != null) {
                    try {
                        out.close();
                    } catch (IOException ex) {
                    }
                }
            }
        }
        return id;
    }
}
