package cz.cuni.amis.pogamut.ut2004.examples.communicatingbot;

import java.util.logging.Level;

import cz.cuni.amis.pogamut.base.agent.module.comm.PogamutJVMComm;
import cz.cuni.amis.pogamut.base.agent.navigation.IPathExecutorState;
import cz.cuni.amis.pogamut.base.agent.navigation.PathExecutorState;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.EventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.base3d.worldview.object.event.WorldObjectDisappearedEvent;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.agent.module.utils.TabooSet;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UT2004PathAutoFixer;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotModuleController;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.BotKilled;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.pogamut.ut2004.utils.UT2004BotRunner;
import cz.cuni.amis.utils.Cooldown;
import cz.cuni.amis.utils.Heatup;
import cz.cuni.amis.utils.collections.MyCollections;
import cz.cuni.amis.utils.exception.PogamutException;
import cz.cuni.amis.utils.flag.FlagListener;

/**
 * CommunicatingBot is featuring utilization of {@link PogamutJVMComm} object to swiftly communicate with
 * other bots within the same JVM ... note that if you're running ALL YOUR bots from single MAIN method,
 * than you're using single JVM == you can easily use {@link PogamutJVMComm#getInstance()} and 
 * {@link PogamutJVMComm#registerAgent(cz.cuni.amis.pogamut.base.agent.IObservingAgent, int)} for inter-agent 
 * communication.
 *
 * Try to execute this bot on DM-1on1-Albatross and wait till they find each other.
 *
 * The bot is based on 02-NAVIGATION-BOT.
 *
 * @author Jimmy
 */
@AgentScoped
public class CommunicatingBot extends UT2004BotModuleController {

	/**
	 * Communication channel used by these bots.
	 * 
	 * Can be 0-MAX_INT, cannot be negative number.
	 */
	protected static final int COMM_CHANNEL = 1;

	/**
     * Taboo set is working as "black-list", that is you might add some
     * NavPoints to it for a certain time, marking them as "unavailable".
     */
    protected TabooSet<NavPoint> tabooNavPoints;
    
    /**
     * Current navigation point we're navigating to.
     */
    protected NavPoint targetNavPoint;
    
    /**
     * Path auto fixer watches for navigation failures and if some navigation
     * link is found to be unwalkable, it removes it from underlying navigation
     * graph.
     *
     * Note that UT2004 navigation graphs are some times VERY stupid or contains
     * VERY HARD TO FOLLOW links...
     */
    protected UT2004PathAutoFixer autoFixer;

    /**
     * The bot is initialized in the environment - a physical representation of
     * the bot is present in the game.
     *
     * @param config information about configuration
     * @param init information about configuration
     */
    @SuppressWarnings("unchecked")
    @Override
    public void botInitialized(GameInfo gameInfo, ConfigChange config, InitedMessage init) {
        // initialize taboo set where we store temporarily unavailable navpoints
        tabooNavPoints = new TabooSet<NavPoint>(bot);

        // auto-removes wrong navigation links between navpoints
        autoFixer = new UT2004PathAutoFixer(bot, pathExecutor, fwMap, aStar, navBuilder);

        // IMPORTANT
        // adds a listener to the path executor for its state changes, it will allow you to 
        // react on stuff like "PATH TARGET REACHED" or "BOT STUCK"
        pathExecutor.getState().addStrongListener(new FlagListener<IPathExecutorState>() {

            @Override
            public void flagChanged(IPathExecutorState changedValue) {
                pathExecutorStateChange(changedValue.getState());
            }
        });
    }

    ///////////////
    //
    // HANDLING COMMUNICATION
    //
    ///////////////
    
    /**
     * First thing you HAVE TO do, is register on {@link PogamutJVMComm} to some channel
     */
    @Override
    public void botFirstSpawn(GameInfo gameInfo, ConfigChange config, InitedMessage init, Self self) {
    	// set log level to FINE to see what is going on
    	PogamutJVMComm.getInstance().getLog().setLevel(Level.FINE);
    	// COMM_CHANNEL == 1 == is channel 
    	// For CTF/team-oriented games you can use self.getTeam()
        PogamutJVMComm.getInstance().registerAgent(bot, COMM_CHANNEL);          
    }
    
    /**
     * DO NOT FORGET TO UNREGISTER YOUR AGENT FROM COMMUNICATION AT THE END!
     * 
     * YOU HAVE TO PROVIDE THIS CLEANUP!!!
     */
    @Override
    public void botShutdown() {
    	 PogamutJVMComm.getInstance().unregisterAgent(bot); 
    }
    
    //
    // FEW BOT CONTROL VARIABLES
    //
    
    /**
     * If this is hot, than I have been spot and I'm searching (turning around) for the bot who has spot me stored within iAmSpotBy.
     */
	protected Heatup iAmSpot = new Heatup(4000);

	/**
	 * This bot has spot me!
	 */
	protected UnrealId iAmSpotBy;

	/**
	 * TRUE == We can see each other! I'm seen by iAmSpotBy and I can see it as well!
	 */
	protected Heatup iAmFixed = new Heatup(5000);
	
	/**
	 * We won't fix/hail the other bot if this is not cool.
	 */
	protected Cooldown doNotFix = new Cooldown(10000);

	/**
	 * Bot I'm fixating/hailing.
	 */
	protected UnrealId iCanSeeId;
	
	/**
	 * How long we're waiting for player to spot us and report it.
	 */
	protected Heatup iCanSee = new Heatup(6000);
    
	/////////
	// COMMUNICATION EVENTS
	/////////
	
	/**
	 * I have received "ICanSeeYou" message !!!
	 * @param event
	 */
    @EventListener(eventClass=YouHaveBeenSpot.class)
    public void iCanSeePlayer(YouHaveBeenSpot event) {
    	if (event.getPlayerId().equals(info.getId())) {
    		return;
    	}
    	if (iAmSpot.isHot()) {
    		// IGNORE, we're already spot by someone
    		return;
    	}
    	// DOES THIS MESSAGE BELONGS TO ME?
    	if (event.getSeePlayerId().equals(info.getId())) {
    		// SOMEBODY CAN SEE ME!
    		iAmSpotBy = event.getPlayerId();
    		iAmSpot.heat();
    		
    		if (players.getVisiblePlayer(iAmSpotBy) != null) {
        		// AND I CAN ALREADY SEE IT AS WELL!
    			iAmFixed.heat();
    			doNotFix.use();
    			body.getCommunication().sendGlobalTextMessage("Hi! I can see you too!");
    			PogamutJVMComm.getInstance().sendToOthers(new YouHaveBeenSpotToo(info.getId(), iAmSpotBy), COMM_CHANNEL, bot);        		
    			// turn clearly to the player we're communicating to
            	move.turnTo(players.getPlayer(iAmSpotBy));
        	}
    	}
    }
    
    /**
     * Somebody can see somebody too!!!
     * @param event
     */
    @EventListener(eventClass=YouHaveBeenSpotToo.class)
    public void iCanSeePlayerToo(YouHaveBeenSpotToo event) {
    	if (iCanSee.isHot()) {
    		// I'm trying to fixate
	    	if (event.getSeePlayerId().equals(info.getId())) {
	       		// And this is bot I'm fixating at!
				iAmFixed.heat();
				doNotFix.use();
				body.getCommunication().sendGlobalTextMessage("That's so cool!");
				PogamutJVMComm.getInstance().sendToOthers(new WeCanSeeEachOther(info.getId(), event.getPlayerId()), COMM_CHANNEL, bot);	    		
	    	}
    	}
    }
    
    /**
     * Cool, we can see each other!
     * @param event
     */
    @EventListener(eventClass=WeCanSeeEachOther.class)
    public void weCanSeeEachOther(WeCanSeeEachOther event) {
    	if (event.getSeePlayerId().equals(info.getId())) {       		
    		body.getCommunication().sendGlobalTextMessage("Hurray!");
    	}
    }
    
    /////////
	// GB2004 EVENTS
	/////////
    
    /**
     * This method is called whenever some PLAYER object appears in my field of view
     * @param event
     */
    @ObjectClassEventListener(eventClass=WorldObjectUpdatedEvent.class, objectClass=Player.class)
    public void playerUpdated(WorldObjectUpdatedEvent<Player> event) {
    	if (info.getId() == null) {
    		// IGNORE EARLY UPDATES, we're not running yet in environment
    		return;
    	}
    	
    	if (!event.getObject().isVisible()) {
    		// react only on VISIBLE PLAYERS
    		return;
    	}
    	
    	if (doNotFix.isHot()) {
    		// forget about it
    		return;
    	}
    	
    	if (iAmSpot.isHot() && iAmSpotBy != null) {
    		// I've been already spot by someone
    		if (iAmSpotBy.equals(event.getObject().getId())) {
    			// And finally I've spot it as well!
    			iCanSeeId = event.getObject().getId();
    			iAmFixed.heat();
    			doNotFix.use();
    			body.getCommunication().sendGlobalTextMessage("Hi! I can see you too!");
    			PogamutJVMComm.getInstance().sendToOthers(new YouHaveBeenSpotToo(info.getId(), iAmSpotBy), COMM_CHANNEL, bot);        		
        		return;
    		}
    	} 
    	
    	// OK, I might try to fix on that player...
    	
		if (iCanSee.isCool()) {
			// I'm not fixating at anybody currently
    		iCanSeeId = event.getObject().getId();
    		iCanSee.heat();
    		body.getCommunication().sendGlobalTextMessage("Hi, " + event.getObject().getName() + "!");
    		PogamutJVMComm.getInstance().sendToOthers(new YouHaveBeenSpot(info.getId(), iCanSeeId), COMM_CHANNEL, bot);    		
    	}
    }
    
    /**
     * This method is called whenever some player disappears from my field of view.
     * @param event
     */
    @ObjectClassEventListener(eventClass=WorldObjectDisappearedEvent.class, objectClass=Player.class)
    public void playerDisappear(WorldObjectDisappearedEvent<Player> event) {
    	if (info.getId() == null) {
    		// IGNORE EARLY UPDATES, we're not running yet in environment
    		return;
    	}
    	if (iCanSeeId != null && iCanSeeId.equals(event.getObject().getId())) {
    		// it was a player I've tried to fix on!
    		iCanSeeId = null;
    		iCanSee.clear();
    		iAmSpot.clear();
    		iAmFixed.clear();
    		body.getCommunication().sendGlobalTextMessage("I've lost you!");
    	}    	
    }
    
    /////////
	// LOGIC
	/////////

    @Override
    public void logic() {
        if (iAmFixed.isHot()) {
        	if (navigation.isNavigating()) {
        		navigation.stopNavigation();
        	}
        	return;
        }
        
        if (iAmSpot.isHot()) {
        	if (navigation.isNavigating()) {
        		navigation.stopNavigation();
        	}
        	// turn around to find who has spot me
        	move.turnHorizontal(30);
        	return;
        }
        
        if (iCanSee.isHot()) {
        	if (navigation.isNavigating()) {
        		navigation.stopNavigation();
        	}
        	// turn clearly to the player we're communicating to
        	move.turnTo(players.getPlayer(iCanSeeId));
        	return;
        }
        
        // no player can be seen
        // => navigate to navpoint
        handleNavPointNavigation();        
    }
    
    //
    // THE REST NAVIGATION STUFF ... not interesting for communication
    //

    private void handleNavPointNavigation() {
        if (navigation.isNavigating()) {
            // WE'RE NAVIGATING TO SOME NAVPOINT
            return;
        }

        // NAVIGATION HAS STOPPED ... 
        // => we need to choose another navpoint to navigate to
        // => possibly follow some players ...

        targetNavPoint = getRandomNavPoint();
        if (targetNavPoint == null) {
            log.severe("COULD NOT CHOOSE ANY NAVIGATION POINT TO RUN TO!!!");
            if (world.getAll(NavPoint.class).size() == 0) {
                log.severe("world.getAll(NavPoint.class).size() == 0, there are no navigation ponits to choose from! Is exporting of nav points enabled in GameBots2004.ini inside UT2004?");
            }
            return;
        }

        navigation.navigate(targetNavPoint);
    }

    /**
     * Called each time our bot die. Good for reseting all bot state dependent
     * variables.
     *
     * @param event
     */
    @Override
    public void botKilled(BotKilled event) {
        navigation.stopNavigation();
    }

    /**
     * Path executor has changed its state (note that {@link UT2004BotModuleController#getPathExecutor()}
     * is internally used by
     * {@link UT2004BotModuleController#getNavigation()} as well!).
     *
     * @param state
     */
    protected void pathExecutorStateChange(PathExecutorState state) {
        switch (state) {
            case PATH_COMPUTATION_FAILED:
                // if path computation fails to whatever reason, just try another navpoint
                // taboo bad navpoint for 3 minutes
                tabooNavPoints.add(targetNavPoint, 180);
                break;

            case TARGET_REACHED:
                // taboo reached navpoint for 3 minutes
                tabooNavPoints.add(targetNavPoint, 180);
                break;

            case STUCK:
                // the bot has stuck! ... target nav point is unavailable currently
                tabooNavPoints.add(targetNavPoint, 60);
                break;

            case STOPPED:
                // path execution has stopped
                targetNavPoint = null;
                break;
        }
    }

    /**
     * Randomly picks some navigation point to head to.
     *
     * @return randomly choosed navpoint
     */
    protected NavPoint getRandomNavPoint() {
        // choose one feasible navpoint (== not belonging to tabooNavPoints) randomly
        NavPoint chosen = MyCollections.getRandomFiltered(getWorldView().getAll(NavPoint.class).values(), tabooNavPoints);

        if (chosen != null) {
            return chosen;
        }

        log.warning("All navpoints are tabooized at this moment, choosing navpoint randomly!");

        // ok, all navpoints have been visited probably, try to pick one at random
        return MyCollections.getRandom(getWorldView().getAll(NavPoint.class).values());
    }

    public static void main(String args[]) throws PogamutException {
        new UT2004BotRunner(CommunicatingBot.class, "CommunicatingBot").setMain(true).setLogLevel(Level.WARNING).startAgents(2);
    }
    
}
