
package cz.cuni.amis.pogamut.ut2004.examples.emotionalbot;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathExecutorState;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UT2004PathAutoFixer;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetector.UT2004PositionStuckDetector;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetector.UT2004TimeStuckDetector;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotModuleController;

import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.*;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.*;

import cz.cuni.amis.pogamut.ut2004.utils.UT2004BotRunner;
import cz.cuni.amis.utils.exception.PogamutException;
import cz.cuni.amis.utils.flag.FlagListener;
import de.affect.emotion.Emotion;
import de.affect.emotion.EmotionType;
import java.text.DecimalFormat;
import java.util.Collection;
import java.util.Random;
import java.util.logging.Level;

/**
 * Example of integration Pogamut with ALMA (http://www.dfki.de/~gebhard/alma/).
 * The bot walks around the map randomly.
 * 
 * Bot is reacting to chat messages ("positive", "negative" and "happy") that
 * generate emotions GRATITUDE ("positive"), ANGER ("negative") and JOY ("happy"). 
 * If the bot will be damaged, emotion anger will be generated.
 * 
 * Bot will start jumping if his mood pleasure is positive and emotion JOY is active.
 * Bot will move slow if his mood pleasure is below 0.
 *
 * @author Michal Bida
 */
@AgentScoped
public class EmotionalBot extends UT2004BotModuleController {

    /** desired name of our agent */
    public String myName = "EmotionalBot";

    /** emotion engine */
    protected ALMA myALMA = null;

    /** event generator */
    protected EEventGenerator myEEventGenerator = null;
    
    /** Helps to delete navigation links we cannot go through */
    private UT2004PathAutoFixer pathAutoFixer;

    /** NavPoint we are going to */
    private NavPoint targetNavPoint = null;
    
    /** current UT2004 time */
    public double currentTime;

    /** If the bot is walking or running */
    public boolean isWalking = false;

    /** Last time we sent message about our current emotional state */
    public double lastMessageTime = 0;

    /** Delay between two emotional messages */
    public static final double MESSAGE_DELAY = 3;

    /**
     * Listener to begin message, that will update currentTime for our agent.
     */
    public IWorldEventListener<BeginMessage> myBegListener = new IWorldEventListener<BeginMessage>() {

        @Override
        public void notify(BeginMessage bm) {
            currentTime = bm.getTime();
        }
    };

    /**
     * Here we can modify initializing command for our bot.
     *
     * @return
     */
    @Override
    public Initialize getInitializeCommand() {
        Initialize init = new Initialize();
        init.setName(myName);
        return init;
    }

    @Override
    public void prepareBot(UT2004Bot bot) {
        pathAutoFixer = new UT2004PathAutoFixer(bot, navigation.getPathExecutor(), fwMap, aStar, navBuilder);
    }

    /**
     * The bot is initilized in the environment - a physical representation of the
     * bot is present in the game.
     *
     * @param config information about configuration
     * @param init information about configuration
     */
    @Override
    public void botInitialized(GameInfo gameInfo, ConfigChange config, InitedMessage init) {

        // initialize emotion engine
        myALMA = new ALMA(this);
        // initialize emotion event generator
        myEEventGenerator = new EEventGenerator(this, myALMA);
        
        //setting log level
        getLog().setLevel(Level.WARNING);

        // Registering listener for updating bot currentTime
        getWorldView().addEventListener(BeginMessage.class, myBegListener);
    }

    /**
     * Main method that controls the bot - makes decisions what to do next.
     * It is called iteratively by Pogamut engine every time a synchronous batch
     * from the environment is received. This is usually 4 times per second - it
     * is affected by visionTime variable, that can be adjusted in GameBots ini file in
     * UT2004/System folder.
     *
     * @throws cz.cuni.amis.pogamut.base.exceptions.PogamutException
     */
    @Override
    public void logic() throws PogamutException {

        // every few seconds we will send a text message with information about our
        // emotional state
        if (currentTime - lastMessageTime > MESSAGE_DELAY) {
            sendMessageAboutEmotions();
            lastMessageTime = currentTime;
        }
        
        // we will look if our mood is bad - if we feel positive we will be running, walking otherwise
        if ((myALMA.getCurrentMood(myName).getPleasure() > 0) && isWalking) {
            isWalking = false;
            getAct().act(new SetWalk().setWalk(false));
        } else if ((myALMA.getCurrentMood(myName).getPleasure() < 0) && !isWalking) {
            isWalking = true;
            getAct().act(new SetWalk().setWalk(true));
        }

        // if we are in a good mood and emotion joy is active, we will be jumping
        if ((myALMA.getDominantEmotion(myName).getType() == EmotionType.Joy) &&
            (myALMA.getDominantEmotion(myName).getIntensity() > 0.45) &&
            (myALMA.getCurrentMood(myName).getPleasure() > 0)) {

            //if we were following a path we will reset
            if (navigation.isNavigating()) {
                navigation.stopNavigation();
            }
            
            getAct().act(new Jump());
            
            return;
        }       

        //if we are not going for an item, we will pick random destination
        if (!navigation.isNavigating()) {
            targetNavPoint = pickNewRandomNavTarget();
            navigation.navigate(targetNavPoint);
            return;
        }        
    }

    /**
     * Constructs and sends text string with information about our mood and dominant emotion.
     */
    private void sendMessageAboutEmotions() {

        String result = "";

        // first check the mood
        if (myALMA.getCurrentMood(myName).getPleasure() > 0)
            result += "I feel happy!";
        else
            result += "I feel sad!";

        // get dominant emotion from ALMA
        Emotion dominantEmotion = myALMA.getDominantEmotion(myName);

        if (dominantEmotion != null)
            result += " My dominant emotion is " + dominantEmotion.getType().toString() +
                " with intensity " + roundTwoDecimals(dominantEmotion.getIntensity()) + "!";
        else
            result += " No dominant emotion at this time.";
        
        getAct().act(new SendMessage().setText(result).setFadeOut(MESSAGE_DELAY).setGlobal(true));
    }


    /**
     * Called each time our bot die. Good for reseting all bot state dependent variables.
     *
     * @param event
     */
    @Override
    public void botKilled(BotKilled event) {
        isWalking = !isWalking;
    }

    /**
     * Randomly picks some navigation point to head to.
     * @return randomly chosen navpoint
     */
    private NavPoint pickNewRandomNavTarget() {
        getLog().severe("Picking new target navpoint.");

        // 1. get all known navigation points
        Collection<NavPoint> navPoints = getWorldView().getAll(NavPoint.class).values();

        // 2. compute index of the target nav point
        int navPointIndex = random.nextInt(navPoints.size());

        // 3. find the corresponding nav point
        int i = 0;
        for (NavPoint nav : navPoints) {
            if (i == navPointIndex) {
                return nav;
            }
            i++;
        }

        // 4. deal with unexpected behavior
        throw new RuntimeException(
                "No navpoint chosen. There are no navpoints in the list of known navpoints");
    }

    public static String roundTwoDecimals(double d) {
        DecimalFormat twoDForm = new DecimalFormat("0.00");
        return twoDForm.format(d);
    }
    
    public static void main(String args[]) throws PogamutException {
    	// wrapped logic for bots executions, suitable to run single bot in single JVM
    	new UT2004BotRunner(EmotionalBot.class, "EmotionalBot").setMain(true).startAgents(1);
    }

}
