/*
 * Copyright (C) 2012 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.spyvsspy.mapgenerator;

import java.awt.Point;
import java.util.HashSet;
import java.util.Map;
import java.util.Random;
import java.util.Set;

/**
 *
 * @author Martin Cerny
 */
public class AbstractRandomMapGenerator {

    Map<ButtonLocation, ButtonDefinition> buttonDefinitions;

    int width;

    int height;

    Random rnd;

    public AbstractRandomMapGenerator(long seed, int width, int height) {
        this.width = width;
        this.height = height;
        rnd = new Random(seed);
    }

    protected int getMaximumNumberOfButtons() {
        //There are two buttons for each door and there are (n - 1)m + n(m-1) doors on the grid
        return 2 * (((width - 1) * height) + (width * (height - 1)));
    }

    protected Point getRandomPoint() {
        return new Point(rnd.nextInt(width), rnd.nextInt(height));
    }

    /**
     * 
     * @param location
     * @param doorStart
     * @param doorEnd
     * @param open
     * @return true if the interaction was added, false if such an interaction (or negative) was already present
     */
    protected boolean addButtonInteractionToLocation(Point location, Point doorStart, Point doorEnd, boolean open) {
        ButtonLocation buttonLocation = getRandomButtonForLocation(location);
        ButtonDefinition butDef = buttonDefinitions.get(buttonLocation);
        if (butDef == null) {
            butDef = new ButtonDefinition(buttonLocation.x, buttonLocation.y, buttonLocation.direction, (Corridor[]) null, null);
            buttonDefinitions.put(buttonLocation, butDef);
        }
        boolean horizontal;
        if (doorStart.x == doorEnd.x) {
            if (Math.abs(doorStart.y - doorEnd.y) != 1) {
                throw new IllegalArgumentException("Start and end are not neighbours");
            }
            horizontal = false;
        } else if (doorStart.y == doorEnd.y) {
            if (Math.abs(doorStart.x - doorEnd.x) != 1) {
                throw new IllegalArgumentException("Start and end are not neighbours");
            }
            horizontal = true;
        } else {
            throw new IllegalArgumentException("Start and end are not neighbours");
        }
        Corridor targetCorridor = new Corridor(Math.min(doorStart.x, doorEnd.x), Math.min(doorStart.y, doorEnd.y), horizontal);
        if (butDef.getOpens().contains(targetCorridor) || butDef.getCloses().contains(targetCorridor)) {
            return false;
        } else {
            if (open) {
                butDef.addOpen(targetCorridor);
            } else {
                butDef.addClose(targetCorridor);
            }
            return true;
        }
    }

    protected ButtonLocation getRandomButtonForLocation(Point location) {
        Direction dir;
        do {
            dir = Direction.values()[rnd.nextInt(Direction.values().length)];
        } while (isButtonOutOfLevel(location, dir));
        return new ButtonLocation(location.x, location.y, dir);
    }

    protected boolean isButtonOutOfLevel(Point location, Direction dir) {
        switch (dir) {
            case EAST: {
                return location.x >= width - 1;
            }
            case NORTH: {
                return location.y <= 0;
            }
            case SOUTH: {
                return location.y >= height - 1;
            }
            case WEST: {
                return location.x <= 0;
            }
            default: {
                throw new IllegalStateException("Unrecognized direction");
            }
        }
    }

    protected Set<Point> getNeighbouringPoints(Point point) {
        Set<Point> result = new HashSet<Point>();
        if (point.x > 0) {
            Point newPoint = new Point(point.x - 1, point.y);
            result.add(newPoint);
        }
        if (point.x < width - 1) {
            Point newPoint = new Point(point.x + 1, point.y);
            result.add(newPoint);
        }
        if (point.y > 0) {
            Point newPoint = new Point(point.x, point.y - 1);
            result.add(newPoint);
        }
        if (point.y < height - 1) {
            Point newPoint = new Point(point.x, point.y + 1);
            result.add(newPoint);
        }
        return result;
    }

    protected class ButtonLocation {

        private int x;

        private int y;

        private Direction direction;

        public ButtonLocation(int x, int y, Direction direction) {
            this.x = x;
            this.y = y;
            this.direction = direction;
        }

        public Direction getDirection() {
            return direction;
        }

        public int getX() {
            return x;
        }

        public int getY() {
            return y;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final ButtonLocation other = (ButtonLocation) obj;
            if (this.x != other.x) {
                return false;
            }
            if (this.y != other.y) {
                return false;
            }
            if (this.direction != other.direction) {
                return false;
            }
            return true;
        }

        @Override
        public int hashCode() {
            int hash = 7;
            hash = 73 * hash + this.x;
            hash = 73 * hash + this.y;
            hash = 73 * hash + (this.direction != null ? this.direction.hashCode() : 0);
            return hash;
        }
    }
}
