/*
 * Copyright (C) 2012 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.spyvsspy.mapgenerator;

import cz.cuni.amis.utils.collections.MyCollections;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 *
 * @author Martin Cerny
 */
public class RandomRelaxedMapGenerator extends AbstractRandomMapGenerator {

    double doorOpenDensity;

    /**
     * 
     * @param width
     * @param height
     * @param doorOpenDensity how many doors can be (on average) opened with one button - this is taken to maximum number of buttons per rectangular level of this size
     */
    public RandomRelaxedMapGenerator(long seed, int width, int height, double doorOpenDensity) {
        super(seed,width, height);
        this.doorOpenDensity = doorOpenDensity;
    }
    
    protected void addPointIfNotReachable(Point point, Set<Point> target, Set<Point> reachablePoints){
        if(!reachablePoints.contains(point)){
            target.add(point);
        }
    }
    
    protected void addNeighbouringUnReachablePoints(Point point, Set<Point> target, Set<Point> reachablePoints){
        for(Point neighbour: getNeighbouringPoints(point)){
            addPointIfNotReachable(neighbour, target, reachablePoints);            
        }
    }
    
    protected Set<Point> getNeighbouringReachablePoints(Point point, Set<Point> reachablePoints){
        Set<Point> result = getNeighbouringPoints(point);
        result.retainAll(reachablePoints);
        return result;
    }
    
    /**
     * Creates a random map without any door closing. A path between 0,0 and width -1, height - 1 will exist (even if it should invalidate {@link #doorOpenDensity}
     * @return 
     */
    public Collection<ButtonDefinition> createRandomMapDefinition(){
        buttonDefinitions = new HashMap<ButtonLocation, ButtonDefinition>();
        
        Set<Point> oneStepReachablePoints = new HashSet<Point>();
        Set<Point> reachablePoints = new HashSet<Point>();
        
/*        for(int i = 0; i < width; i++){
            for(int j = 0; j < height ; j++){
                unreachablePoints.add(new Point(i, j));                
            }
        }
 
        unreachablePoints.remove(new Point(0, 0));
 
 */
        reachablePoints.add(new Point(0, 0));
        addNeighbouringUnReachablePoints(new Point(0,0), oneStepReachablePoints, reachablePoints);
        
        Point targetPoint = new Point(width - 1, height - 1);
        
        int doorOpenCount = 0;
        int desiredOpenDoorCount = (int) (getMaximumNumberOfButtons() * doorOpenDensity);
        
        //while target is not reachable, choose random neighbouring location 
        while(!reachablePoints.contains(targetPoint)){
            Point nextReachable = MyCollections.getRandom(oneStepReachablePoints);
            Point reachFrom = MyCollections.getRandom(getNeighbouringReachablePoints(nextReachable, reachablePoints));
            Point buttonLocation = MyCollections.getRandom(reachablePoints);
            addButtonInteractionToLocation(buttonLocation, reachFrom, nextReachable, true);
            
            oneStepReachablePoints.remove(nextReachable);            
            reachablePoints.add(nextReachable);
            addNeighbouringUnReachablePoints(nextReachable, oneStepReachablePoints, reachablePoints);
            doorOpenCount++;
        }
        
        System.out.println("Final guaranteed path length: " + doorOpenCount);
        
        if(doorOpenCount > desiredOpenDoorCount){
            double newDensity = ((double)doorOpenCount) / getMaximumNumberOfButtons();
            System.out.println("Needed more than desired number of interactions to reach the target. The doorOpenDensity is " + newDensity + ", should have been " + doorOpenDensity);
        }
        
        //Just add random interactions until desired density is achieved
        while(doorOpenCount < desiredOpenDoorCount){
            
            boolean addedNew = false;
            while (!addedNew){
                Point randomTo = getRandomPoint();
                Point randomFrom = MyCollections.getRandom(getNeighbouringPoints(randomTo));
                Point buttonLocation = MyCollections.getRandom(reachablePoints);
                reachablePoints.add(randomTo);
                addedNew = addButtonInteractionToLocation(buttonLocation, randomFrom, randomTo, true);
                //iterate until we find an interaction that was not added - this should not be very difficult
            }
            doorOpenCount++;            
        }
        
        return buttonDefinitions.values();
    }
}
