/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package cz.cuni.amis.pogamut.spyvsspy.mapgenerator;

import com.thoughtworks.xstream.XStream;
import cz.cuni.amis.pogamut.spyvsspy.metadata.Button;
import cz.cuni.amis.pogamut.spyvsspy.metadata.Door;
import cz.cuni.amis.pogamut.spyvsspy.metadata.Room;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.DefaultT3dGenerator;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.IT3dGenerator;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.T3dElementHelper;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.datatypes.KismetLinkTarget;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.datatypes.Point3D;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.datatypes.Rotation3D;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.AbstractUnrealActor;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.kismet.Sequence;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.map.NavigationPoint;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.map.PathNode;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.map.PlayerStart;
import cz.cuni.amis.pogamut.unreal.t3dgenerator.elements.map.Trigger;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author Martin Cerny
 */
public class Generator {
    public final String BUTTON1_PATHNODE_SUFFIX = "_Button1";
    public final String BUTTON1_TRIGGER_SUFFIX = "_Button1Trigger";
    public final String BUTTON2_PATHNODE_SUFFIX = "_Button2";
    public final String BUTTON2_TRIGGER_SUFFIX = "_Button2Trigger";

    Iterable<ButtonDefinition> buttonDefinitions;

    int[][] horizontalCorridorPresence;
    int[][] verticalCorridorPresence;
    int levelWidth;
    int levelHeight;
    
    boolean generatePlayerStarts;

    public Generator(Iterable<ButtonDefinition> buttonDefinitions, boolean generatePlayerStarts) {

        this.generatePlayerStarts = generatePlayerStarts;
        this.buttonDefinitions = buttonDefinitions;

        levelWidth = 0;
        levelHeight = 0;

        for (ButtonDefinition buttonDef : buttonDefinitions) {
            for (Corridor c : buttonDef.getAllCorridor()) {
                if (c.getDestinationX() + 1 > levelWidth) {
                    levelWidth = c.getDestinationX() + 1;
                }
                if (c.getDestinationY() + 1 > levelHeight) {
                    levelHeight = c.getDestinationY() + 1;
                }
            }
        }

        this.horizontalCorridorPresence = new int[levelWidth - 1][levelHeight];
        this.verticalCorridorPresence = new int[levelWidth][levelHeight - 1];

        for (ButtonDefinition buttonDef : buttonDefinitions) {
            boolean buttonOutOfLevel = false;
            if (buttonDef.getLocX() < 0 || buttonDef.getLocY() < 0 || buttonDef.getLocX() >= levelWidth || buttonDef.getLocY() >= levelHeight) {
                buttonOutOfLevel = true;
            } else {
                switch (buttonDef.getDirection()) {
                    case NORTH: {
                        if (buttonDef.getLocY() <= 0) {
                            buttonOutOfLevel = true;
                        } else {
                            verticalCorridorPresence[buttonDef.getLocX()][buttonDef.getLocY() - 1] = 1;
                        }                        
                        break;
                    }
                    case WEST: {
                        if (buttonDef.getLocX() <= 0) {
                            buttonOutOfLevel = true;
                        } else {
                            horizontalCorridorPresence[buttonDef.getLocX() - 1][buttonDef.getLocY()] = 1;
                        }                        
                        break;
                    }
                    case SOUTH: {
                        if (buttonDef.getLocY() >= levelHeight - 1) {
                            buttonOutOfLevel = true;
                        } else {
                            verticalCorridorPresence[buttonDef.getLocX()][buttonDef.getLocY()] = 1;
                        }                        
                        break;
                    }
                    case EAST: {
                        if (buttonDef.getLocX() >= levelWidth -1) {
                            buttonOutOfLevel = true;
                        } else {
                            horizontalCorridorPresence[buttonDef.getLocX()][buttonDef.getLocY()] = 1;
                        }                        
                        break;
                    }
                }
            }

            if(buttonOutOfLevel){
                throw new IllegalArgumentException("Inconsistent definition. Button at " + buttonDef.getLocX() + "," + buttonDef.getLocY() + " " + buttonDef.getDirection() + " is out of level bounds");
            }
            for (Corridor c : buttonDef.getAllCorridor()) {
                if (c.isHorizontal()) {
                    horizontalCorridorPresence[c.getOriginX()][c.getOriginY()] = 1;
                } else {
                    verticalCorridorPresence[c.getOriginX()][c.getOriginY()] = 1;
                }
            }
        }
    }

    public String getRoomName(int x, int y) {
        return "" + (char) ('A' + x) + y;
    }

    public String getCorridorName(int startX, int startY, int endX, int endY) {
        String smer = "";
        if (startX == endX) {
            smer = "_Vert";
        } else if (startY == endY) {
            smer = "_Horiz";
        }
        return "Corridor_" + getRoomName(startX, startY) + "_" + getRoomName(endX, endY) + smer;
    }

    protected int getNumSurrounded(int x, int y) {
        int numSurround = 0;
        for (Direction d : Direction.values()) {
            if (isRoomSorrounded(x, y, d)) {
                numSurround++;
            }
        }
        return numSurround;
    }

    protected boolean isRoomSorrounded(int x, int y, Direction dir) {
        switch (dir) {
            case NORTH: {
                if (y <= 0) {
                    return false;
                }
                return verticalCorridorPresence[x][y - 1] != 0;
            }
            case EAST: {
                if (x >= levelWidth - 1) {
                    return false;
                }
                return horizontalCorridorPresence[x][y] != 0;
            }
            case SOUTH: {
                if (y >= levelHeight - 1) {
                    return false;
                }
                return verticalCorridorPresence[x][y] != 0;
            }
            case WEST: {
                if (x <= 0) {
                    return false;
                }
                return horizontalCorridorPresence[x - 1][y] != 0;
            }
            default: {
                throw new IllegalArgumentException("Unrecognized direction");
            }
        }
    }


    public void generate(File t3dFile, File metadataFile, File kismetFile) {
        List<AbstractUnrealActor> actors = new ArrayList<AbstractUnrealActor>();

        //For some reason WorldInfo is causing trouble in importing the level to editor
//        WorldInfo worldInfo = new WorldInfo();
//        worldInfo.setKillZ(-100f);
//        actors.add(worldInfo);

        KismetGenerator kismetGenerator = new KismetGenerator(this);

        int gridSize = 9 * 256;

        PrefabDefinition corner = new PrefabDefinition(new Point3D(256, 0, 0), "SpyVsSpy.Corner");
        PrefabDefinition threeCross = new PrefabDefinition(new Point3D(368 - 112, 0, 0), "SpyVsSpy.3Cross");
        PrefabDefinition fourCross = new PrefabDefinition(new Point3D(-116 + 368, 0, 0), "SpyVsSpy.4Cross");
        PrefabDefinition straight = new PrefabDefinition(new Point3D(368 - 112, 0, 0), "SpyVsSpy.Straight");
        PrefabDefinition deadEnd = new PrefabDefinition(new Point3D(256, 0, 0), "SpyVsSpy.DeadEnd");

        PrefabDefinition corridor = new PrefabDefinition(new Point3D(0, -128, 0), "SpyVsSpy.Corridor");


        PrefabDefinition[][] roomsDefinitions = new PrefabDefinition[levelWidth][levelHeight];
        int[][] roomsRotations = new int[levelWidth][levelHeight];


        for (int x = 0; x < levelWidth; x++) {
            for (int y = 0; y < levelHeight; y++) {
                switch (getNumSurrounded(x, y)) {
                    case 0: {
                        roomsDefinitions[x][y] = null;
                        break;
                    }
                    case 1: {
                        roomsDefinitions[x][y] = deadEnd;
                        for (Direction d : Direction.values()) {
                            if (isRoomSorrounded(x, y, d)) {
                                roomsRotations[x][y] = d.getYawRotation() + 1; //+1 is a specfic coefficent for deadEnd. (default deadEnd heads west)
                                break;
                            }
                        }
                        break;
                    }
                    case 2: {
                        if (isRoomSorrounded(x, y, Direction.NORTH) == isRoomSorrounded(x, y, Direction.SOUTH)) {
                            roomsDefinitions[x][y] = straight;
                            if (isRoomSorrounded(x, y, Direction.NORTH)) {
                                roomsRotations[x][y] = 1;
                            } else {
                                roomsRotations[x][y] = 0;
                            }
                        } else {
                            roomsDefinitions[x][y] = corner;
                            //Rotation 0 corner is sourrounded NORTH and WEST
                            if (isRoomSorrounded(x, y, Direction.NORTH)) {
                                if (isRoomSorrounded(x, y, Direction.EAST)) {
                                    roomsRotations[x][y] = 1;
                                } else { //this means NORTH-WEST
                                    roomsRotations[x][y] = 0;
                                }
                            } else {
                                if (isRoomSorrounded(x, y, Direction.EAST)) { //this means SOUTH-EAST
                                    roomsRotations[x][y] = 2;
                                } else { //this means SOUTH-WEST
                                    roomsRotations[x][y] = -1;
                                }
                            }
                        }
                        break;
                    }
                    case 3: {
                        roomsDefinitions[x][y] = threeCross;
                        for (Direction dir : Direction.values()) {
                            if (!isRoomSorrounded(x, y, dir)) {
                                //rotaion zero 3cross is not sorrounded SOUTH
                                roomsRotations[x][y] = dir.getYawRotation() + 2;
                            }
                        }
                        break;
                    }
                    case 4: {
                        roomsDefinitions[x][y] = fourCross;
                        break;
                    }
                }
            }
        }


        Room[][] locationMetaData = new Room[levelWidth][levelHeight];


        for (int i = 0; i < levelWidth; i++) {
            for (int j = 0; j < levelHeight; j++) {
                if (roomsDefinitions[i][j] != null) {
                    Point3D location = new Point3D(-j * gridSize, i * gridSize, 0);
                    Rotation3D rotation = new Rotation3D(0, roomsRotations[i][j] * 16384, 0);
                    String name = "Room_" + getRoomName(i, j);
                    actors.add(roomsDefinitions[i][j].createPrefabInstance(location, rotation, name));

                    NavigationPoint navigationPoint;
                    Point3D navigationPointLocation = location.add(new Point3D(0, 0, 25));
                    if(generatePlayerStarts){
                        navigationPoint = new PlayerStart(navigationPointLocation);
                    } else {
                        navigationPoint = new PathNode(navigationPointLocation);
                    }
                    navigationPoint.setName("PlayerStart_" + getRoomName(i, j));
                    actors.add(navigationPoint);

                    locationMetaData[i][j] = new Room(name, navigationPoint.getName());
                } else {
                    locationMetaData[i][j] = null;
                }
            }
        }

        Point3D button1TriggerLocation = new Point3D(-75, -640, 96);
        float buttonTriggerCollisionHeight = 80;
        float buttonTriggerCollisionRadius = 128;
        Point3D button1PathNodeLocation = new Point3D(-15,-640, 25);
        
        List<Door> doorMetadata = new ArrayList<Door>();
        Sequence[][] horizontalCorridorSequences = new Sequence[levelWidth - 1][levelHeight];

        for (int i = 0; i < horizontalCorridorPresence.length; i++) {
            for (int j = 0; j < horizontalCorridorPresence[i].length; j++) {
                if (horizontalCorridorPresence[i][j] != 0) {
                    Point3D location = new Point3D(-j * gridSize, i * gridSize + (gridSize / 2), 0);
                    String name = getCorridorName(i, j, i + 1, j);
                    actors.add(corridor.createPrefabInstance(location, new Rotation3D(0, 0, 0), name));
                    Door door = new Door(locationMetaData[i][j], locationMetaData[i + 1][j]);
                    doorMetadata.add(door);
                    horizontalCorridorSequences[i][j] = kismetGenerator.addCorridorSequence(name, door.getFrom().getRoomName(), door.getTo().getRoomName(), i * 2 + 1, j * 2);

                    PathNode button1Node = new PathNode(location.add(button1PathNodeLocation));
                    button1Node.setNameForReferences(name + BUTTON1_PATHNODE_SUFFIX);
                    actors.add(button1Node);
                    
                    Trigger button1Trigger = new Trigger(location.add(button1TriggerLocation),buttonTriggerCollisionHeight, buttonTriggerCollisionRadius);
                    button1Trigger.setNameForReferences(name+ BUTTON1_TRIGGER_SUFFIX);
                    actors.add(button1Trigger);

                    PathNode button2Node = new PathNode(location.add(button1PathNodeLocation.negateXandY()));
                    button2Node.setNameForReferences(name + BUTTON2_PATHNODE_SUFFIX);
                    actors.add(button2Node);
                    
                    Trigger button2Trigger = new Trigger(location.add(button1TriggerLocation.negateXandY()),buttonTriggerCollisionHeight, buttonTriggerCollisionRadius);
                    button2Trigger.setNameForReferences(name + BUTTON2_TRIGGER_SUFFIX);
                    actors.add(button2Trigger);
                }
            }
        }

        Sequence[][] verticalCorridorSequences = new Sequence[levelWidth][levelHeight - 1];

        for (int i = 0; i < verticalCorridorPresence.length; i++) {
            for (int j = 0; j < verticalCorridorPresence[i].length; j++) {
                if (verticalCorridorPresence[i][j] != 0) {
                    Point3D location = new Point3D(-j * gridSize - (gridSize / 2), i * gridSize, 0);
                    String name = getCorridorName(i, j, i, j + 1);
                    actors.add(corridor.createPrefabInstance(location, new Rotation3D(0, 16384, 0), name));
                    Door door = new Door(locationMetaData[i][j], locationMetaData[i][j + 1]);
                    doorMetadata.add(door);
                    verticalCorridorSequences[i][j] = kismetGenerator.addCorridorSequence(name,door.getFrom().getRoomName(), door.getTo().getRoomName(), i * 2, j * 2 + 1);

                    PathNode button1Node = new PathNode(location.add(button1PathNodeLocation.switchXandY().negateX()));
                    button1Node.setNameForReferences(name + BUTTON1_PATHNODE_SUFFIX);
                    actors.add(button1Node);
                    
                    Trigger button1Trigger = new Trigger(location.add(button1TriggerLocation.switchXandY().negateX()),buttonTriggerCollisionHeight, buttonTriggerCollisionRadius);
                    button1Trigger.setNameForReferences(name+ BUTTON1_TRIGGER_SUFFIX);
                    actors.add(button1Trigger);
                    

                    PathNode button2Node = new PathNode(location.add(button1PathNodeLocation.switchXandY().negateY()));
                    button2Node.setNameForReferences(name + BUTTON2_PATHNODE_SUFFIX);
                    actors.add(button2Node);

                    Trigger button2Trigger = new Trigger(location.add(button1TriggerLocation.switchXandY().negateY()),buttonTriggerCollisionHeight, buttonTriggerCollisionRadius);
                    button2Trigger.setNameForReferences(name+ BUTTON2_TRIGGER_SUFFIX);
                    actors.add(button2Trigger);

                }
            }
        }

        List<Button> buttonMetadata = new ArrayList();

        //Creating connections between buttons and opening / closing of doors
        for(ButtonDefinition def : buttonDefinitions){
            Sequence sourceSequence;
            int buttonId;
            String corridorName;

            switch(def.getDirection()){
                case NORTH: {
                    sourceSequence = verticalCorridorSequences[def.getLocX()][def.getLocY() - 1];
                    buttonId = 2;
                    corridorName = getCorridorName(def.getLocX(), def.getLocY()- 1, def.getLocX(), def.getLocY());
                    break;
                }
                case EAST: {
                    sourceSequence = horizontalCorridorSequences[def.getLocX()][def.getLocY()];
                    buttonId = 1;
                    corridorName = getCorridorName(def.getLocX(), def.getLocY(), def.getLocX() + 1, def.getLocY());
                    break;
                }
                case SOUTH: {
                    sourceSequence = verticalCorridorSequences[def.getLocX()][def.getLocY()];
                    buttonId = 1;
                    corridorName = getCorridorName(def.getLocX(), def.getLocY(), def.getLocX(), def.getLocY() + 1);
                    break;
                }
                case WEST: {
                    sourceSequence = horizontalCorridorSequences[def.getLocX() - 1][def.getLocY()];
                    buttonId = 2;
                    corridorName = getCorridorName(def.getLocX() - 1, def.getLocY(), def.getLocX(), def.getLocY());
                    break;
                }
                default: {
                    throw new IllegalStateException("Unrecognized value of direction");
                }
            }
            
            if(sourceSequence == null){
                throw new IllegalArgumentException("No corridor found for button: " + def);
            }

            String sourceLabel;
            String pathNodeName;
            String triggerName;
            
            sourceLabel = "Button" + buttonId;
            if(buttonId == 1){
                  pathNodeName = corridorName + BUTTON1_PATHNODE_SUFFIX;
                  triggerName = corridorName + BUTTON1_TRIGGER_SUFFIX;
            }
            else if(buttonId == 2){
                  pathNodeName = corridorName + BUTTON2_PATHNODE_SUFFIX;
                  triggerName = corridorName + BUTTON2_TRIGGER_SUFFIX;
            } else {
                throw new IllegalStateException();
            }


            List<Door> opensDoor = new ArrayList<Door>();
            List<Door> closesDoor = new ArrayList<Door>();

            for(Corridor c : def.getOpens()){
                Sequence targetSequence;
                targetSequence = findTargetSequence(c, horizontalCorridorSequences, verticalCorridorSequences);
                sourceSequence.addOutputLinkTarget(sourceLabel, new KismetLinkTarget(targetSequence, "Open"));
                opensDoor.add(new Door(locationMetaData[c.getOriginX()][c.getOriginY()], locationMetaData[c.getDestinationX()][c.getDestinationY()]));
            }
            for(Corridor c : def.getCloses()){
                Sequence targetSequence;
                targetSequence = findTargetSequence(c, horizontalCorridorSequences, verticalCorridorSequences);
                sourceSequence.addOutputLinkTarget(sourceLabel, new KismetLinkTarget(targetSequence, "Close"));
                closesDoor.add(new Door(locationMetaData[c.getOriginX()][c.getOriginY()], locationMetaData[c.getDestinationX()][c.getDestinationY()]));
            }

            buttonMetadata.add(new Button(pathNodeName,locationMetaData[def.getLocX()][def.getLocY()], opensDoor, closesDoor, triggerName));
        }

        try {
            OutputStreamWriter out = new FileWriter(t3dFile);
            IT3dGenerator gen = new DefaultT3dGenerator();
            gen.generateT3d(T3dElementHelper.wrapActorsIntoMap("Gen_Test", actors), out);
            out.close();
        } catch (IOException e) {
            e.printStackTrace();
        }

        kismetGenerator.saveToFile(kismetFile);

        List<Object> metaDataList = new ArrayList<Object>();
        for (int i = 0; i < levelWidth; i++) {
            for (int j = 0; j < levelHeight; j++) {
                if (locationMetaData[i][j] != null) {
                    metaDataList.add(locationMetaData[i][j]);
                }
            }
        }
        metaDataList.addAll(doorMetadata);
        metaDataList.addAll(buttonMetadata);

        try {
            OutputStreamWriter out = new FileWriter(metadataFile);
            XStream xstream = new XStream();
            out.write(xstream.toXML(metaDataList));
//            ObjectOutputStream outputStream = xstream.createObjectOutputStream(out);
//            outputStream.writeObject(metaDataList);
//            outputStream.close();
            out.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private Sequence findTargetSequence(Corridor c, Sequence[][] horizontalCorridorSequences, Sequence[][] verticalCorridorSequences) {
        Sequence targetSequence;
        if (c.isHorizontal()) {
            targetSequence = horizontalCorridorSequences[c.getOriginX()][c.getOriginY()];
        } else {
            targetSequence = verticalCorridorSequences[c.getOriginX()][c.getOriginY()];
        }
        return targetSequence;
    }
}
