package cz.cuni.amis.pogamut.udk.agent.navigation;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathFuture;
import cz.cuni.amis.pogamut.base.agent.navigation.impl.PathFuture;
import cz.cuni.amis.pogamut.base.communication.worldview.react.EventReact;
import cz.cuni.amis.pogamut.base.communication.worldview.react.EventReactOnce;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.GetPath;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.udk.communication.translator.shared.events.Path;
import cz.cuni.amis.utils.future.FutureStatus;

/**
 * {@link IPathFuture} implementation that is using UDK inner AStar algorithm for finding the path inside UDK
 * environment. 
 * <p><p>
 * <b>WARNING:</b> UDK has a limition set on the path length. It will return only the first
 * 16 navpoints that are leading to the path's target. Whenever path executor happens to tell you, that
 * the target is reached, you should compare your bot current location with {@link UDKAStarPathFuture#getPathTo()}.
 * <p><p>
 * Note that the path that is produced by this future contains mix of {@link NavPoint} and {@link Location} objects.
 * Usually {@link Location} objects are only the first and last elements of the path and the rest are {@link NavPoint}s.
 * 
 * @author Jimmy
 */
public class UDKAStarPathFuture extends PathFuture<ILocated> {

	private static final int PATH_TIMEOUT = 10;

	private static Object idMutex = new Object();
	
	private static long lastId = 0;
	
	private String pathId;
	
	private EventReactOnce<Path> pathReaction;
	
	private EventReact<EndMessage> endReaction;

	private IVisionWorldView worldView;
	
	private Logger log;
	
	private Double startTime;

	public UDKAStarPathFuture(UDKBot bot, ILocated pathFrom, ILocated pathTo) {
		super(pathFrom, pathTo, bot.getEventBus(), bot.getWorldView());
		log = bot.getLogger().getCategory(this.getClass().getSimpleName());
		synchronized(idMutex) {
			pathId = "UDKAStarPathFuture_" + (++lastId);
		}
		pathReaction = new EventReactOnce<Path>(Path.class, bot.getWorldView()){
			@Override
			protected void react(Path event) {
				if (pathId.equals(event.getPathId())) {
					eventPath(event);
				}
			}			
		};
		endReaction = new EventReact<EndMessage>(EndMessage.class, bot.getWorldView()) {
			@Override
			protected void react(EndMessage event) {
				eventEndMessage(event);
			}		
		};
		log.finer("Requesting path from '" + pathFrom + "' to '" + pathTo + "' under id '" + pathId + "'.");
		bot.getAct().act(new GetPath().setLocation(pathTo.getLocation()).setId(pathId));
		log.fine("Path requested, listening for the result (timeout " + PATH_TIMEOUT + "s)");		
		worldView = bot.getWorldView();
	}

	@Override
	protected boolean cancelComputation(boolean mayInterruptIfRunning) {
		pathReaction.disable();
		endReaction.disable();
		return getStatus() == FutureStatus.FUTURE_IS_BEING_COMPUTED;
	}
	
	protected void eventEndMessage(EndMessage event) {
		if (startTime == null) startTime = event.getTime();
		if (event.getTime() - startTime > PATH_TIMEOUT) {
			pathReaction.disable();
			endReaction.disable();
			if (getStatus() == FutureStatus.FUTURE_IS_BEING_COMPUTED) {
				computationException(new UDKAStarPathTimeoutException("Path did not came from GB2004 in " + PATH_TIMEOUT + "s.", log, this));
			}
		}
	}

	protected void eventPath(Path event) {
		endReaction.disable();
		List<ILocated> result = new ArrayList<ILocated>(event.getPath().size());
		for (int i = 0; i < event.getPath().size(); ++i) {			
			UnrealId routeId = event.getPath().get(i).getRouteId();
			NavPoint nav = (NavPoint) worldView.get(routeId);
			if (nav == null) {
				result.add(event.getPath().get(i).getLocation());
			} else {
				result.add(nav);
			}
		}
		setResult(result);
	}

}
