package cz.cuni.amis.pogamut.udk.utils;

import java.util.List;
import java.util.logging.Level;

import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IAgentParameters;
import cz.cuni.amis.pogamut.base.agent.utils.runner.IAgentDescriptor;
import cz.cuni.amis.pogamut.base.agent.utils.runner.impl.MultipleAgentRunner;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.factory.IAgentFactory;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.PogamutProperty;
import cz.cuni.amis.pogamut.udk.agent.params.UDKAgentParameters;
import cz.cuni.amis.pogamut.udk.bot.IUDKBot;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotFactory;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotModule;
import cz.cuni.amis.utils.NullCheck;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * This class has only one purpose - execute ONE OR MORE BOTS inside 'main' method. You can't use it for anything else!
 * It is THE SHORTCUT of all SHORTCUTS to execute multiple bots, wait till they finishe and close the whole Pogamut.
 * <p><p>
 * Designed especially for the usage inside NetBeans projects.
 * <p><p>
 * NOTE: by default, all bots get paused after they start and they are resumed after all bots are present in UDK. To
 * change this behaviour pass 'false' through {@link MultipleUDKBotRunner#setPausing(boolean)}. 
 * <p><p>
 * NOTE: It's not even meant to be instantiated twice for two different batch of bots and consequently executed in two different threads!
 * Single-purpose class only ;-)
 * <p><p>
 * NOTE: It might be very interesting for you to check out the source of method {@link MultipleUDKBotRunner#startAgent()} to
 * see how the agent should be instantiated via {@link UDKBotFactory} using {@link UDKBotModule}.
 * <p><p>
 * 
 * 
 * @author Jimmy
 */
public class MultipleUDKBotRunner<BOT extends UDKBot, PARAMS extends UDKAgentParameters, MODULE extends UDKBotModule> extends MultipleAgentRunner<BOT, PARAMS, MODULE> {
	
	/**
	 * Default host where the instances are going to be connected as defaults, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
    protected String host;
    
    /**
	 * Default port where the instances are going to be connected as defaults, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
    protected int port;
    
    /**
	 * Default name that will serve as a basis for {@link IAgentId}, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
	protected String name;
	
	/** 
	 * Construct the runner + specify all defaults.
	 * 
 	 * @param name default name that serve as a basis for {@link IAgentId}
	 * @param host default host where the instances are going to be connected
	 * @param port default port where the instances are going to be connected
	 */
	public MultipleUDKBotRunner(String name, String host, int port) {
        this.name = name;
        this.port = port;
        this.host = host;
    }

	/**
	 * Construct the runner + specify the default name, host:port will be taken from the Pogamut platform properties.
	 * 
	 * @param factory factory to be used for creating new {@link IUDKBot} instances
	 * @param name default name that serve as a basis for {@link IAgentId}
	 */
    public MultipleUDKBotRunner(String name) {
        this(
        	name, 
        	Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_HOST.getKey()), 
        	Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_PORT.getKey())
        );
    }
    
    /**
     * Returns name that is going to be used to form new {@link IAgentId} of the bots.
     *     
     * @return name used for the newly started bots
     */
    public String getName() {
		return name;
	}

    /**
     * Sets name that is going to be used to form new {@link IAgentId} of the bots.
     * <p><p>
     * If null is passed, generic "UDKBot" will be set.
     *     
     * @param name name used for the newly started bots
     * @return this instance
     */
	public MultipleUDKBotRunner<BOT, PARAMS, MODULE> setName(String name) {
		if (name == null) name = "UDKBot";
		this.name = name;
		return this;
	}

	/**
     * Returns host, where newly launched bots will be connected to.
     * 
     * @return host running GBUDK server
     */
    public String getHost() {
		return host;
	}

    /**
     * Sets host, where newly launched bots will be connected to.
     * 
     * @param host host running GBUDK server (can't be null)
     * @return this instance
     */
	public MultipleUDKBotRunner<BOT, PARAMS, MODULE> setHost(String host) {
		this.host = host;
		NullCheck.check(this.host, "host");
		return this;
	}

	/**
     * Returns port, where newly launched bots will be connected to.
     * 
     * @return port at the host where GBUDK server is listening for bot connections
     */
	public int getPort() {
		return port;
	}

	/**
     * Sets port, where newly launched bots will be connected to.
     * 
     * @param port at the host where GBUDK server is listening for bot connections
     * @return this instance
     */
	public MultipleUDKBotRunner<BOT, PARAMS, MODULE> setPort(int port) {
		this.port = port;
		return this;
	}
	
	/**
     * We're setting the logging level to {@link Level#WARNING} here so the bot won't log much.
     */
    @Override
    protected void preStartHook(BOT agent) throws PogamutException {
        agent.getLogger().setLevel(Level.WARNING);
    }

    /**
     * Provides default parameters that is, {@link IAgentId} using {@link MultipleUDKBotRunner#name} and {@link SocketConnectionAddress}
     * using {@link MultipleUDKBotRunner#host} and {@link MultipleUDKBotRunner#port}.
     */
	@Override
	protected IAgentParameters newDefaultAgentParameters() {
		return new UDKAgentParameters().setAgentId(new AgentId(name)).setWorldAddress(new SocketConnectionAddress(host, port));
	}

	/**
	 * Uses {@link UDKBotFactory} for agent construction.
	 */
	@Override
	protected IAgentFactory newAgentFactory(MODULE agentModule) {
		return new UDKBotFactory<IUDKBot, UDKAgentParameters>(agentModule);
	}
	
	public List<BOT> startAgents(IAgentDescriptor<PARAMS,MODULE>... agentDescriptors) {
		return super.startAgents(agentDescriptors);
	};

}
