package cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient;

import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute.IAttributedReplication;

/** Ingredient implementation component
 * 
 * Updates cooking progress stored in attributes.
 */
public class IngredientComponent {
	
	public static final float MAX_MAGNITUDE = 4;
	
	protected static final String CHARREDNESS_ATTR = "charredness";
	protected static final String BOILEDNESS_ATTR = "boiledness";
	protected static final String TOP_FRIEDNESS_ATTR = "topFriedness";
	protected static final String BOTTOM_FRIEDNESS_ATTR = "bottomFriedness";
	protected static final String IS_UPSIDE_DOWN_ATTR = "isUpsideDown";
	
	protected float boilingBaseTemperature;
	protected float boilingStandardTemperature;
	protected float boilednessPerMagnitudePerSecond;
	
	protected float charringBaseTemperature;
	protected float charringStandardTemperature;
	protected float charrednessPerMagnitudePerSecond;
	protected float charrednessResistance;
	protected float charrednessResistanceStirringBonus;
	protected float maxBufferedCharredness;
	
	protected float fryingBaseTemperature;
	protected float fryingStandardTemperature;
	protected float friednessPerMagnitudePerSecond;
	
	protected IAttributedReplication attributeProvider;
		
	/** Constructor
	 * 
	 * @param initAttributeProvider attribute provider
	 */
	public IngredientComponent( IAttributedReplication initAttributeProvider ) {
		attributeProvider = initAttributeProvider;
		
		boilednessPerMagnitudePerSecond = 0;
		charrednessPerMagnitudePerSecond = 0;
		friednessPerMagnitudePerSecond = 0;
	}
	
	/** Initialize boiling behavior
	 * 
	 * @param initBoilingBaseTemperature temperature in degrees of celsius up to which boiling process has magnitude equal to zero
	 * @param initBoilingStandardTemperature temperature in degrees of celsius at which boiling process has magnitude equal to one
	 * @param initBoilednessPerMagnitudePerSecond how much boiledness per second per magnitude is gained, linear process
	 */
	public void initializeBoilingBehavior(
		float initBoilingBaseTemperature,
		float initBoilingStandardTemperature,
		float initBoilednessPerMagnitudePerSecond
	) {
		boilingBaseTemperature = initBoilingBaseTemperature;
		boilingStandardTemperature = initBoilingStandardTemperature;
		boilednessPerMagnitudePerSecond = initBoilednessPerMagnitudePerSecond;
	}
	
	/** Initialize charring behavior
	 * 
	 * @param initCharringBaseTemperature temperature in degrees of celsius up to which charring process has magnitude equal to zero
	 * @param initCharringStandardTemperature temperature in degrees of celsius at which charring process has magnitude equal to one
	 * @param initCharrednessPerMagnitudePerSecond how much charring per second per magnitude is gained, linear process
	 * @param initCharrednessResistance how much charring is resisted per second
	 * @param initCharrednessResistanceStirringBonus how much charring is additionally resisted per second of stirring
	 * @param initMaxBufferedCharredness how much charredness is buffered
	 */
	public void initializeCharringBehavior(
		float initCharringBaseTemperature,
		float initCharringStandardTemperature,
		float initCharrednessPerMagnitudePerSecond,
		float initCharrednessResistance,
		float initCharrednessResistanceStirringBonus,
		float initMaxBufferedCharredness
	) {
		charringBaseTemperature = initCharringBaseTemperature;
		charringStandardTemperature = initCharringStandardTemperature;
		charrednessPerMagnitudePerSecond = initCharrednessPerMagnitudePerSecond;
		charrednessResistance = initCharrednessResistance;
		charrednessResistanceStirringBonus = initCharrednessResistanceStirringBonus;
		maxBufferedCharredness = initMaxBufferedCharredness;
	}
	
	/** Initialize frying behavior
	 * 
	 * @param initFryingBaseTemperature temperature in degrees of celsius up to which frying process has magnitude equal to zero
	 * @param initFryingStandardTemperature temperature in degrees of celsius at which frying process has magnitude equal to one
	 * @param initFriednessPerMagnitudePerSecond friedness per magnitude per second, 0.0 to use boiling behavior instead
	 */
	public void initializeFryingBehavior(
		float initFryingBaseTemperature,
		float initFryingStandardTemperature,
		float initFriednessPerMagnitudePerSecond 
	) {
		fryingBaseTemperature = initFryingBaseTemperature;
		fryingStandardTemperature = initFryingStandardTemperature;
		friednessPerMagnitudePerSecond = initFriednessPerMagnitudePerSecond;
	}
	
	/** See IIngredient
	 */
	public float getCharredness() {
		return attributeProvider.getAttributes().floats().get( CHARREDNESS_ATTR );
	}
	
	/** See IIngredient
	 */
	public float getBoiledness() {
		return attributeProvider.getAttributes().floats().get( BOILEDNESS_ATTR );
	}
	
	/** See IIngredient
	 */
	public float getBottomFriedness() {
		return attributeProvider.getAttributes().floats().get( BOTTOM_FRIEDNESS_ATTR );
	}
	
	/** See IIngredient
	 */
	public float getTopFriedness() {
		return attributeProvider.getAttributes().floats().get( TOP_FRIEDNESS_ATTR );
	}
	
	/** See IIngredient
	 */
	public boolean getIsUpsideDown() {
		return attributeProvider.getAttributes().bools().get( IS_UPSIDE_DOWN_ATTR );
	}
	
	/* Cooking tick as implemented in unreal script source:
 	
	function cookingTick( float temperature, float heatConductivity, bool isStirred, float deltaTime )
	{
		local float magnitude, friednessGrowth;
		
		if ( boilednessPerMagnitudePerSecond != 0.0 )
		{
			magnitude = computeMagnitude( temperature, boilingBaseTemperature, boilingStandardTemperature );
			
			setBoiledness( FMin( 1.0, getBoiledness()+boilednessPerMagnitudePerSecond*magnitude*deltaTime*heatConductivity ) );
		}
		
		if ( charrednessPerMagnitudePerSecond != 0.0 )
		{
			magnitude = computeMagnitude( temperature, charringBaseTemperature, charringStandardTemperature );
			
			bufferedCharredness += (
				( 
					charrednessPerMagnitudePerSecond * magnitude / heatConductivity 
					- 
					( charrednessResistance + ( isStirred ? charrednessResistanceStirringBonus : 0.0 ) )
				)
				*
				deltaTime
			);
			
			if ( bufferedCharredness > maxBufferedCharredness )
			{
				setCharredness( FMin( 1.0, getCharredness()+bufferedCharredness-maxBufferedCharredness ) );
				bufferedCharredness = maxBufferedCharredness;
			}
			else if ( bufferedCharredness < 0 )
			{
				bufferedCharredness = 0;
			}
		}
			
		if ( friednessPerMagnitudePerSecond != 0.0 )
		{
			magnitude = computeMagnitude( temperature, fryingBaseTemperature, fryingStandardTemperature );
			
			friednessGrowth = charrednessPerMagnitudePerSecond*magnitude*deltaTime;
	
			if ( getIsUpsideDown() )
			{
				setTopFriedness( FMin( 1.0, getTopFriedness() + friednessGrowth ) );
			}
			else
			{
				setBottomFriedness( FMin( 1.0, getBottomFriedness() + friednessGrowth ) );
			}
		}
	}
	
	*/
	
	/** Compute magnitude
	 * 
	 * Server-side only.
	 */
	public float computeMagnitude( float temperature, float baseTemperature, float standardTemperature )
	{
		float squareRoot;
		if ( temperature <= baseTemperature )
		{
			return 0.0f;
		}
		else
		{
			squareRoot = (temperature-baseTemperature) / ( standardTemperature - baseTemperature );
			return Math.min( MAX_MAGNITUDE, squareRoot*squareRoot );
		}
	}
}