package cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute;

import java.util.Iterator;

import cz.cuni.amis.pogamut.emohawk.agent.module.attribute.AttributeUpstream;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.ListMapReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.ListMapEntryReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.object.SpecializedClass;

/** Abstract attribute view
 * 
 * Provides a set of low level operations with type-filtering logic.
 * 
 * @author Paletz
 */
public class AttributeRawView<Attribute extends IAssignableAttributeReplica<IntermediateValue>, IntermediateValue> {

	protected SpecializedClass<Attribute> attributeClass;
	protected AttributeUpstream upstream;
	protected ListMapReplica<IAttributeReplica> attributes;
	
	/** Constructor
	 */
	public AttributeRawView(
		ListMapReplica<IAttributeReplica> attributes,
		AttributeUpstream upstream,
		SpecializedClass<Attribute> attributeClass
	) {
		this.attributes = attributes;
		this.upstream = upstream;
		this.attributeClass = attributeClass;
	}
	
	/** Get attributes
	 */
	public ListMapReplica<IAttributeReplica> getAttributes() {
		return attributes;
	}
	
	/** Get attribute record
	 *
	 * @param attributeName attribute name, must exist and have the right type
	 * @return attribute record
	 */
	public Attribute getAttributeRecord( String attributeName ) {
		@SuppressWarnings("unchecked")
		Attribute attribute = (Attribute) attributes.get( attributeName );
		assert( attributeClass.isInstance( attribute ) );
		
		return attribute;
	}
	
	/** Set remote attribute value
	 * 
	 * @param attributeName attribute name
	 * @param value value to set
	 */
	public void send( String attributeName, IntermediateValue value )
	{
		Attribute attribute = getAttributeRecord( attributeName );
		
		assert( attribute.getAccess() == AttributeAccess.ATTRIBUTE_ACCESS_WRITE );
		assert(
			attribute.getOwnerControllerId() == IAttributeReplica.OWNER_COMMUNITY
			||
			attribute.getOwnerControllerId() == upstream.getControllerId()
		);
		
		attribute.serialize( upstream.getOutputStream(), value );
		upstream.setAttribute( attribute.getReplicationId() );	
	}
	
	/** Tell if attribute exists and has the right type
	 *
	 * @param attributeName attribute name
	 * @param access required access
	 * @return true if attribute exists and has the right type and access, false otherwise
	 */
	public boolean exists( String attributeName, AttributeAccess access )
	{
		if ( attributes.containsKey( attributeName ) ) {
			IAttributeReplica attribute = attributes.get( attributeName );
			return attributeClass.isInstance( attribute ) && attribute.getAccess() == access;
		} else {
			return false;
		}
	}
	
	/** Iterate over attributes in view
	 * 
	 * @return view iterator
	 */
	public Iterator<ListMapEntryReplica<Attribute>> iterator() {
		final Iterator<ListMapEntryReplica<IAttributeReplica>> iterator = attributes.iterator();
		
		return new Iterator<ListMapEntryReplica<Attribute>>() {
			
			ListMapEntryReplica<Attribute> next;
			
			@Override
			public boolean hasNext() {
				fetchNext();
				return next != null; // attribute can't be null
			}
			
			@Override
			public ListMapEntryReplica<Attribute> next() {
				fetchNext();
				assert( next != null ); // attribute can't be null
				return next;
			}
			
			@Override
			public void remove() {
				throw new UnsupportedOperationException( "Can't remove attribute." );
			}
			
			@SuppressWarnings("unchecked")
			protected void fetchNext() {
				if ( next != null ) {
					return;
				}
				
				while ( iterator.hasNext() ) {
					ListMapEntryReplica<IAttributeReplica> candidate = iterator.next();
					if ( attributeClass.isInstance( candidate.getValue() ) ) {
						next = new ListMapEntryReplica<Attribute>( candidate.getKey(), (Attribute) candidate.getValue() );
						return;
					}
				}
			}
		};
	}
}