package cz.cuni.amis.pogamut.emohawk.agent.module.replication;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ControllerInfoReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.PawnReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.action.ActionRegistryReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute.AttributeManagerReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute.IAttributeReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute.foggyref.FoggyRefAttributeReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute.foggyreflist.FoggyRefListAttributeReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute.primitive.PrimitiveAttributeReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.attribute.primitivelist.ListAttributeReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.FoggyRefReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.ListMapReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.ListMapEntryReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.PrimitiveBoxReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.sll.SinglyLinkedListReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.sll.SinglyLinkedListCoreReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ds.sll.SinglyLinkedListNodeReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.object.IGenericObjectReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.object.IObjectReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.object.ReplicationId;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.object.SpecializedClass;
import cz.cuni.amis.pogamut.emohawk.agent.module.stream.StreamManager;
import cz.cuni.amis.pogamut.emohawk.communication.replication.ObjectReplication;
import cz.cuni.amis.pogamut.emohawk.communication.replication.ObjectTearOff;
import cz.cuni.amis.pogamut.emohawk.communication.replication.ObjectUpdate;
import cz.cuni.amis.pogamut.emohawk.communication.stream.PayloadType;
import cz.cuni.amis.pogamut.ut2004.bot.IUT2004Bot;

/** Unreal object replication client
 * 
 * Implements java side of object replication.
 * <p>
 * Implementation of some replicated objects may require additional modules. 
 * Such modules may be registered before replication is initialized via {@link #registerModule} and later obtained by {@link #getModule}.
 * <p>
 * <b>Implementation details</b>
 * <p>
 * Replication in java is a bit more complicated because Unreal classes must be mapped to Java classes.
 * A class map exists for that purpose.
 * <p>
 * This is further complicated by templated unreal classes. Because templated class can access it's type parameter in the runtime it 
 * can't be mapped to a simple generic class. However, the type parameter can be stored manually, see {@link IGenericObjectReplication}. 
 * <p>
 * Because type parameter may be such generic replicable class too, it is not sufficient to store type parameter as a Class<?>. 
 * Type parameter of type parameter woulld be lost. The {@link SpecializedClass} is defined to work around this limitation.
 * 
 * @author Paletz
 *
 */
public class ObjectReplicationClient implements IReplicatedObjectTracker {

	protected static String LETTER = "[a-zA-Z]";
	protected static String WORD = "(?:"+LETTER+"+)";
	protected static String RAW_PARAMETER = "(?:(?:"+LETTER+"|(?:__))+)";
	protected static Pattern GENERIC_CLASS_NAME_PATTERN = Pattern.compile( WORD+"(?:_"+RAW_PARAMETER+")*" );
	protected static Pattern GENERIC_CLASS_TYPE_PARAMETER_PATTERN = Pattern.compile(RAW_PARAMETER);
	
	protected HashMap<String,Class<?>> classMap;
	protected HashMap<Integer,IObjectReplication> objectMap;
	protected StreamManager streamManager;
	protected HashMap<Class<?>,Object> modules;
	
	/** Constructor
	 * 
	 * @param bot bot
	 * @param streamManager manager of stream between pogamut and the Unreal server
	 */
	public ObjectReplicationClient( IUT2004Bot bot, StreamManager streamManager ) {
		this.objectMap = new HashMap<Integer,IObjectReplication>();
		this.classMap = new HashMap<String,Class<?>>();
		this.streamManager = streamManager;
		this.modules = new HashMap<Class<?>,Object>();
		
		bot.getWorldView().addEventListener(
			ObjectReplication.class,
			objectReplicationListener
		);
		bot.getWorldView().addEventListener(
			ObjectUpdate.class,
			objectUpdateListener
		);
		bot.getWorldView().addEventListener(
			ObjectTearOff.class,
			objectTearOffListener
		);
		
		classMap.put( "int", Integer.class );
		classMap.put( "bool", Boolean.class );
		classMap.put( "float", Float.class );
		classMap.put( "string", String.class );
		classMap.put( "EhIReplicableObject", IObjectReplication.class );
		classMap.put( "EhPawn", PawnReplication.class );
		classMap.put( "EhIAttribute", IAttributeReplication.class );
		classMap.put( "EhSinglyLinkedList", SinglyLinkedListReplication.class );
		classMap.put( "EhSinglyLinkedListCore", SinglyLinkedListCoreReplication.class );
		classMap.put( "EhSinglyLinkedListNode", SinglyLinkedListNodeReplication.class );
		classMap.put( "EhAttributeManager", AttributeManagerReplication.class );
		classMap.put( "EhListMap", ListMapReplication.class );
		classMap.put( "EhListMapEntry", ListMapEntryReplication.class );
		classMap.put( "EhPrimitiveBox", PrimitiveBoxReplication.class );
		classMap.put( "EhFoggyRef", FoggyRefReplication.class );
		classMap.put( "EhPrimitiveAttribute", PrimitiveAttributeReplication.class );
		classMap.put( "EhListAttribute", ListAttributeReplication.class );
		classMap.put( "EhFoggyRefAttribute", FoggyRefAttributeReplication.class );
		classMap.put( "EhFoggyRefListAttribute", FoggyRefListAttributeReplication.class );
		classMap.put( "EhControllerInfo", ControllerInfoReplication.class );
		classMap.put( "EhActionRegistry", ActionRegistryReplication.class );
	}
	
	@Override
	public IObjectReplication getObject( int replicationId )
	{
		assert( replicationId >= 0 );
		assert( objectMap.containsKey( replicationId ) );
		return objectMap.get(replicationId);
	}
	
	@Override
	public boolean exists( int replicationId )
	{
		assert( replicationId >= 0 );
		
		return objectMap.containsKey( replicationId );
	}
	
	/** Register module to be available to replicated objets
	 * 
	 * Only one module can be registered per class. Repeated registration is an error.
	 * 
	 * @param moduleClass module class or interface, acts as a key
	 * @param module the registered module
	 */
	public <T,M extends T> void registerModule( Class<T> moduleClass, M module ) {
		assert( !modules.containsKey( moduleClass ) );
		modules.put( moduleClass, module );
	}
	
	/** Obtain previously registered module
	 * 
	 * @param moduleClass module class or interface, acts as a key
	 * @return module registered to the key
	 */
	public <T> T getModule( Class<T> moduleClass ) {
		assert( modules.containsKey( moduleClass ) );
		@SuppressWarnings("unchecked")
		T retval = (T) modules.get( moduleClass );
		return retval;
	}
	
	/** Replicate object.
	 * 
	 * Result of replication call originating from the server.
	 * 
	 * @param replicationId replication ID the newly replicated object has to assume
	 * @param objectClass class of the object 
	 */
	protected IWorldEventListener<ObjectReplication> objectReplicationListener = new IWorldEventListener<ObjectReplication>() {

		@Override
		public void notify(ObjectReplication event) {
			ReplicationId replicationId = event.getReplicationId();
			String objectClass = event.getObjectClass();
			
			assert( ! exists( replicationId.getIndex() ) );
			SpecializedClass<?> specializedClass = findClass( objectClass );
			assert( IObjectReplication.class.isAssignableFrom( specializedClass.getGenericClass() ) );
			
			IObjectReplication object;
			try {
				object = (IObjectReplication) specializedClass.getGenericClass().newInstance();
			} catch (InstantiationException e) {
				throw new RuntimeException( e );
			} catch (IllegalAccessException e) {
				throw new RuntimeException( e );
			}
			
			if ( specializedClass.getGenericClass().getTypeParameters().length > 0 ) {
				IGenericObjectReplication genericObject = (IGenericObjectReplication) object;
				for ( int i=0; i<specializedClass.getGenericClass().getTypeParameters().length; ++i ) {
					genericObject.setTypeParameter( i, specializedClass.getTypeParameter( i ) );
				}
			}
			
			object.initializeImage( ObjectReplicationClient.this, replicationId );
			objectMap.put( replicationId.getIndex(), object );
		}
	};

	/** Find class by name.
	 * 
	 * Result of replication call originating from the server.
	 * 
	 * @param objectClass Unreal script class name.
	 * @return specialized class ( class + type parameters if it is a generic class ) 
	 */
	protected SpecializedClass<?> findClass( String objectClass )
	{
		if ( ! GENERIC_CLASS_NAME_PATTERN.matcher( objectClass ).matches() ) {
			throw new RuntimeException( "Invalid class name \""+objectClass+"\"." );
		}
		
		List<String> typeParameters = new LinkedList<String>();
		String className;
		if ( objectClass.contains("_") ) {
			className = objectClass.substring( 0, objectClass.indexOf( "_" ) );
			
			String parameterString = objectClass.substring( className.length()+1 );
			Matcher parameterMatcher = GENERIC_CLASS_TYPE_PARAMETER_PATTERN.matcher( parameterString );
			while ( parameterMatcher.find() ) {
				String parameter = parameterMatcher.group(0);
				typeParameters.add( parameter.replace( "__", "_" ) );
			}
		} else {
			className = objectClass;
		}
		
		if ( !classMap.containsKey(className) ) {
			throw new RuntimeException( "Can't replicate unknown class \""+objectClass+"\"." );
		}
		
		Class<?> genericClass = classMap.get(className);
				
		if ( genericClass.getTypeParameters().length != typeParameters.size() ) {
			throw new RuntimeException( "Incorrect number of type parameters for \""+className+"\" ( "+typeParameters+" )." );
		}
		
		LinkedList<SpecializedClass<?>> resolvedTypeParameters = new LinkedList<SpecializedClass<?>>();
		for ( String typeParameter : typeParameters ) {
			resolvedTypeParameters.add( findClass(typeParameter) );
		}
		
		@SuppressWarnings({ "unchecked", "rawtypes" })
		SpecializedClass<?> retval = new SpecializedClass<Object>( (Class) genericClass, resolvedTypeParameters );
		
		return retval;
	}

	/** Tear-off object.
	 * 
	 * Internal. Result of tear-off call originating from the server.
	 * 
	 * @param replicationId replication ID the object to tear-off
	 * @param expectedClass error detection
	 */
	protected IWorldEventListener<ObjectTearOff> objectTearOffListener = new IWorldEventListener<ObjectTearOff>() {
		
		@Override
		public void notify( ObjectTearOff event ) {
			
			int replicationIndex = event.getReplicationIndex();
			
			assert( exists( replicationIndex ) );
			
			IObjectReplication object = objectMap.get(replicationIndex);
			
			objectMap.remove( replicationIndex );
			
			object.finalizeReplication();
		}
	};

	/** Update object.
	 * 
	 * Internal. Result of update call originating from the server.
	 * 
	 * @param replicationId replication ID the object to update
	 * @param inputStream input stream with update data
	 * @param expectedClass error detection
	 */
	protected IWorldEventListener<ObjectUpdate> objectUpdateListener = new IWorldEventListener<ObjectUpdate>() {

		@Override
		public void notify(ObjectUpdate event) {
			int replicationIndex = event.getReplicationIndex();
						
			IObjectReplication object = getObject(replicationIndex);
			
			object.receive( streamManager.getInputStream() );
			assert( streamManager.getInputStream().tellNext() == PayloadType.PAYLOAD_TYPE_EOF );
			streamManager.clearInput();
		}
	};
}