package cz.cuni.amis.pogamut.emohawk.agent.module.essence;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.ControllerInfoReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.PawnReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.action.ActionRegistryReplication;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.essence.IEssenceReplication;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;

/** Map of associations of replicable objects
 * 
 * This map allows pogamut-emohawk to associate essences with world objects without the need to modify existing pogamut-ut2004 info messages.
 * For example, ReplicableObject in Unreal script may send name of the associated world object and Java receive() can associate the name with the essence.
 * 
 * @author Paletz
 */
public class EssenceMap implements IEssenceMap {

	protected IVisionWorldView worldView;
	protected HashMap<String,PawnReplication> pawns; 	// bot ID (unreal script Controller.name) -> pawn 
	protected HashMap<String,ControllerInfoReplication> controllers; 	// bot ID (unreal script Controller.name) -> controller 
	protected ActionRegistryReplication actionRegistry;
	protected HashMap<String,IEssenceReplication> essences; // entity ID (unreal script Object name) -> essence

	public EssenceMap( IVisionWorldView worldView ) {
		this.worldView = worldView;
		pawns = new HashMap<String,PawnReplication>();
		controllers = new HashMap<String,ControllerInfoReplication>();
		essences = new HashMap<String,IEssenceReplication>();
	}
	
	/** Associate Pawn object with a bot ID
	 * 
	 * @param botId bot ID
	 * @param pawn Pawn object
	 */
	public void associatePawn( String botId, PawnReplication pawn ) {
		pawns.put( botId, pawn );
	}
	
	@Override
	public PawnReplication retrievePawn( String botId ) {
		return pawns.get( botId );
	}
	
	@Override
	public PawnReplication retrievePawn(IWorldObject worldObject) {
		if ( worldObject instanceof Self ) {
			return retrievePawn( ((Self)worldObject).getBotId().getStringId() );
		} else {
			return retrievePawn( worldObject.getId().getStringId() );
		}
	}

	/** Associate Controller object with a bot ID
	 * 
	 * @param botId bot ID
	 * @param controller Controller object
	 */
	public void associateController( String botId, ControllerInfoReplication controller ) {
		controllers.put( botId, controller );
	}
	
	@Override
	public ControllerInfoReplication retrieveController( String botId ) {
		return controllers.get( botId );
	}
	
	@Override
	public ControllerInfoReplication retrieveController(IWorldObject worldObject) {
		return retrieveController( worldObject.getId().getStringId() );
	}
	
	/** Associate action registry
	 * 
	 * @param actionRegistry action registry
	 */
	public void associateActionRegistry( ActionRegistryReplication actionRegistry ) {
		this.actionRegistry = actionRegistry;
	}
	
	@Override
	public ActionRegistryReplication retrieveActionRegistry() {
		return actionRegistry;
	}
		
	/** Associate essence to world object ID
	 *
	 * @param worldObjectId world object ID
	 * @param essence
	 */
	public void associateEssence( String worldObjectId, IEssenceReplication essence ) {
		essences.put( worldObjectId, essence );
	}

	@Override
	public IEssenceReplication retrieveEssence(String worldObjectId) {
		return essences.get( worldObjectId );
	}
	
	@Override
	public IEssenceReplication retrieveEssence( IWorldObject entity ) {
		return retrieveEssence( entity.getId().getStringId() );
	}

	@Override
	public <Essence extends IEssenceReplication> Map<IWorldObject, Essence> getAllVisible( Class<Essence> essenceClass ) {
    	HashMap<IWorldObject,Essence> retval = new HashMap<IWorldObject,Essence>();
    	for ( Entry<WorldObjectId,IViewable> entry : worldView.getVisible().entrySet() ) {
    		IWorldObject worldObject = worldView.get( entry.getKey() );
    		IEssenceReplication essence = retrieveEssence( entry.getValue() );
    		if ( essence == null ) {
    			essence = retrievePawn( worldObject );
    		}
    		if ( essence != null && essenceClass.isAssignableFrom( essence.getClass() ) ) {
    			@SuppressWarnings("unchecked")
				Essence castedEssence = (Essence) essence;
    			retval.put( worldObject, castedEssence );
    		}
    	}
    	return retval;
	}
}
