package cz.cuni.amis.pogamut.base.communication.worldview;

import java.util.Map;

import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.AnnotationListenerRegistrator;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.EventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectDestroyedEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.base.component.bus.IComponentBus;
import cz.cuni.amis.pogamut.base3d.worldview.object.event.WorldObjectAppearedEvent;
import cz.cuni.amis.pogamut.base3d.worldview.object.event.WorldObjectDisappearedEvent;

/**
 * Interface for the world view.
 * <p><p>
 * It assumes that each world description consists of two entites:
 * <ol>
 * <li>actual events described by ({@link IWorldEvent})</li>
 * <li>existing objects described by ({@link IWorldObject})</li>
 * </ol>
 * The world view is designed to accept {@link IWorldEvent} and {@link IWorldObjectEvent} which describes
 * the changes in the world. User of the world view is allowed to set up listeners for various events as well as objects.  
 * <p>
 * Therefore there are two types of listeners:
 * <ol>
 * <li>{@link IWorldEventListener} - we will refer to them as 'event' listeners</li>
 * <li>{@link IWorldObjectEventListener} - we will refer to them as 'object' listeners<br/>
 *      note that there are also a descendant {@link IWorldObjectListener} that is easier to use.</li>
 * </ol>
 * Generally, the world view provides 5 methods user may use to hook up the listener of a various level of specificity.
 * <p><p>
 * <ul>
 * <li>
 * <b>Level A listeners ({@link IWorldView#addEventListener(Class, IWorldEventListener)}):</b> Primarily - these listeners can be hooked up according to the event class, for instance there is {@link WorldObjectAppearedEvent} that
 * implements IWorldObjectEvent. Therefore, if you want to be notified about every object that appears in the
 * field of view of the agent, you should register an implementor of {@link IWorldObjectEventListener} to the class {@link WorldObjectAppearedEvent}.class.
 * <p><p>
 * The motivation here is, that the class of event is unique identifier of the event and therefore it may serve
 * as identification for listener registration.
 * <p><p>
 * Also notice that {@link IWorldObjectEvent} is extending {@link IWorldEvent} (of course!), therefore by registering a listener to {@link IWorldEvent}.class
 * will allow you to sniff every events that will happen in the world (or generated by the world view).
 * </li>
 * </ul>
 * <p><p>
 * Furthermore - there are {@link IWorldObjectEventListener}s that can be registered to receive only events that are happening on some 
 * {@link IWorldObject}. These are:
 * <ul>
 * <li><b>Level B listeners ({@link IWorldView#addObjectListener(Class, IWorldObjectListener)}):</b> listen to all events on a specific class of objects by specifying the class (this is different from the class of the event because
 * this time we're querying class of the object not the class of the event)</li>
 * <li><b>Level C listeners ({@link IWorldView#addObjectListener(Class, Class, IWorldObjectListener)}):</b> listen to specific event on specific class of objects</li>
 * <li><b>Level D listeners ({@link IWorldView#addObjectListener(WorldObjectId, IWorldObjectListener)}):</b> listen to all events on specific object via specifying its object id</li>
 * <li><b>Level E listeners ({@link IWorldView#addObjectListener(WorldObjectId, Class, IWorldObjectListener)}):</b> listen to a specific event on the specific object (via object id)</li>
 * </ul>
 * <p>
 * The listener notifying scheme follows the rule from "general" listeners
 * to "specific" listeners. The listeners are invoked in this order:
 * <ol>
 * <li>Level A listeners ({@link EventListener})</li>
 * <li>Level B listeners ({@link ObjectClassListener})</li>
 * <li>Level C listeners ({@link ObjectClassEventListener})</li>
 * <li>Level D listeners ({@link ObjectListener})</li>
 * <li>Level E listeners ({@link ObjectEventListener})</li>
 * </ol>
 * <p><p>
 * When you need to listen on various events, use method annotations (see the links in the list) and {@link AnnotationListenerRegistrator} 
 * to automatically hook up listeners for you - note that the {@link AnnotationListenerRegistrator} are currently not supporting inheritance.
 *
 * @author Jimmy
 */
public interface IWorldView extends IWorldChangeEventInput {

	// =========
	// COMPONENT
	// =========
	
	public IComponentBus getEventBus();
	
	// ======
	// EVENTS
	// ======

	/**
	 * Adds listener to a specific event (Level A listeners). Note that the event listener must be able
	 * to handle events of the class 'event'.
	 * <p><p>
	 * It is the most general type of listener-registration allowing you to sniff any type of events.
	 * <p><p>
	 * Events passed to the listener are filtered only according to the 'event' class.
	 * <p><p>
	 * <b>WARNING:</b> even though the method does not require templated class and listener, you must be sure
	 * that 'listener' can accept 'eventClass'.
	 *
	 * @param eventClass which event types you want to receive
	 * @param listener where you want to handle these events
	 */
	public void addEventListener(Class<?> eventClass, IWorldEventListener<?> listener);
	
	/**
	 * Adds listener to all events that happens on any object of the 'objectClass' (Level B listeners).
	 * <p><p>
	 * Events passed to the listener are filtered according to the 'objectClass'.
	 * <p><p>
	 * <b>WARNING:</b> even though the method does not require templated classes and listener, you must be sure
	 * that 'listener' accepts all events (IWorldObjectEvent) for objects of 'objectClass'.
	 *
	 * @param objectClass which object class you want to listen at
	 * @param eventClass which event class you want to receive
	 * @param listener where you want to handle these events
	 */
	public void addObjectListener(Class<?> objectClass, IWorldObjectEventListener<?, ?> listener);

	/**
	 * Adds listener to a specified 'event' that occurs on the specific 'objectClass' (Level C listeners).
	 * <p><p>
	 * Events passed to the listener are filtered according to the 'event' and 'objectClass' of the object the event happened upon.
	 * <p><p>
	 * <b>WARNING:</b> even though the method does not require templated classes and listener, you must be sure
	 * that 'listener' accepts 'eventClass' for objects of 'objectClass'.
	 *
	 * <p>
	 * eventClass can be any implementor of {@link IWorldObjectEvent}. E.g.
	 * {@link WorldObjectAppearedEvent}, {@link WorldObjectDestroyedEvent}, {@link WorldObjectDisappearedEvent},
	 * {@link WorldObjectFirstEncounteredEvent} or {@link WorldObjectUpdatedEvent}.
	 * </p>
	 *
	 * @param objectClass which object class you want to listen at
	 * @param eventClass which event class you want to receive
	 * @param listener where you want to handle these events
	 */
	public void addObjectListener(Class<?> objectClass, Class<?> eventClass, IWorldObjectEventListener<?,?> listener);

	/**
	 * Adds listener to all events that happens on object with specific 'objectId' (Level D listeners).
	 * <p><p>
	 * Events passed to the listener are filtered according to the 'objectId' of the object.
	 * <p><p>
	 * <b>WARNING:</b> you must ensure that 'listener' can accept the event raised on object of specified 'objectId'.
	 *
	 * @param objectId which object you want to listen at
	 * @param listener where you want to handle events
	 */
	public void addObjectListener(WorldObjectId objectId, IWorldObjectEventListener<?, ?> listener);

	/**
	 * Adds listener to a specified 'event' that occurs on the specific object with 'objectId' (Level E listeners).
	 * <p><p>
	 * Events passed to the listener are filtered according to the 'event' and 'objectId' of the object.
	 * <p><p>
	 * <b>WARNING:</b> even though the method does not require templated classes and listener, you must be sure
	 * that 'listener' accepts 'eventClass' for objects of specified 'objectId'.
	 *
	 * @param objectId which object you want to listen at
	 * @param eventClass which event classes you want to receive
	 * @param listener where you want to handle these events
	 */
	public void addObjectListener(WorldObjectId objectId, Class<?> eventClass, IWorldObjectEventListener<?, ?> listener);

	/**
	 * Removes listener from a specific event (Level A listeners).
	 *
	 * @param eventClass which events class you want to remove the listener from
	 * @param listener you want to remove
	 */
	public void removeEventListener(Class<?> eventClass, IWorldEventListener<?> listener);

	/**
	 * Removes listener from specific 'objectClass' listening for specified 'event' (Level B listeners).
	 *
	 * @param objectClass class of objects you want the listener to remove from
	 * @param eventClass which events class you want to remove the listener from
	 * @param listener you want to remove
	 */
	public void removeObjectListener(Class<?> objectClass, IWorldObjectEventListener<?,?> listener);
	
	/**
	 * Removes listener from specific 'objectClass' listening for specified 'event' (Level C listeners).
	 *
	 * @param objectClass class of objects you want the listener to remove from
	 * @param eventClass which events class you want to remove the listener from
	 * @param listener you want to remove
	 */
	public void removeObjectListener(Class<?> objectClass, Class<?> eventClass, IWorldObjectEventListener<?,?> listener);


	/**
	 * Removes listener from objects with specified 'objectId' (Level D Listeners).
	 *
	 * @param objectId id of object you want the listener to remove from
	 * @param listener you want to remove
	 */
	public void removeObjectListener(WorldObjectId objectId, IWorldObjectEventListener<?, ?> listener);

	/**
	 * Removes listener to a specified 'event' that occurs on the specific object with 'objectId' (Level E listeners).
	 *
	 * @param objectId id of object you want the listener to remove from
	 * @param eventClass event class you want to stop receiving in the listener
	 * @param listener you want to remove
	 */
	public void removeObjectListener(WorldObjectId objectId, Class<?> eventClass, IWorldObjectEventListener<?, ?> listener);

	/**
	 * Removes listener from every listeners category (from every listener level).
	 * <p><p>
	 * <b>WARNING:</b> Can be time consuming! (Iterating through all levels of listeners.)
	 *
	 * @param listener you want to remove from all listener levels
	 */
	public void removeListener(IWorldEventListener<?> listener);

	/**
	 * Tests whether the 'listener' is listening to a specific event (Level A listeners).
	 *
	 * @param eventClass which events you want to receive
	 * @param listener that is tested
	 * @return whether the listener is listening
	 */
	public boolean isListening(Class<?> eventClass, IWorldEventListener<?> listener);

	/**
	 * Tests whether the 'listener' is listening at specified 'objectClass' (Level B listeners).
	 *
	 * @param objectClass where the listener is tested
	 * @param listener that is tested
	 * @return whether the listener is listening
	 */
	public boolean isListening(Class<?> objectClass, IWorldObjectEventListener<?,?> listener);	
	
	/**
	 * Tests whether the 'listener' is listening at specified 'objectClass' for specified 'event' (Level C listeners).
	 *
	 * @param objectClass where the listener is tested
	 * @param eventClass where the listener is tested
	 * @param listener that is tested
	 * @return whether the listener is listening
	 */
	public boolean isListening(Class<?> objectClass, Class<?> eventClass, IWorldObjectEventListener<?,?> listener);


	/**
	 * Tests whether the 'listener' is listening at specified 'objectId' (Level D Listeners).
	 *
	 * @param objectId where the listener is tested
	 * @param listener that is tested
	 * @return whether the listener is listening
	 */
	public boolean isListening(WorldObjectId objectId, IWorldObjectEventListener<?, ?> listener);

	/**
	 * Tests whether the 'listener' is listening to a specified 'event' that occurs on the specific object with 'objectId' (Level E listeners).
	 *
	 * @param objectId where the listener is tested
	 * @param eventClass what class is tested
	 * @param listener that is tested
	 * @return whether the listener is listening
	 */
	public boolean isListening(WorldObjectId objectId, Class<?> eventClass, IWorldObjectEventListener<?, ?> listener);

	/**
	 * Checks whether this listener is hooked to the world view (at any listener level).
	 * <p><p>
	 * <b>WARNING:</b> Can be time consuming! (Iterating through all levels of listeners.)
	 *
	 * @param listener
	 * @return
	 */
	public boolean isListening(IWorldEventListener<?> listener);

	// =======
	// OBJECTS
	// =======

	/**
	 * Returns map of all objects that are present in the world view.
	 * <p><p>
	 * <b>WARNING:</b> If you will do iteration over the map, you must synchronize on it.
	 */
	@SuppressWarnings("unchecked")
	public Map<Class, Map<WorldObjectId, IWorldObject>> getAll();
	
	/**
	 * Returns map of all objects of a specific type that are present in the world view.
 	 * <p><p>
	 * <b>WARNING:</b> If you will do iteration over the map, you must synchronize on it.
	 *
	 * @param type
	 * @return
	 */
	public <T extends IWorldObject> Map<WorldObjectId, T> getAll(Class<T> type);
	
	/**
     * Returns the only instance of required object if present, if there are
     * more instances of this object then {@link IllegalArgumentException} will be thrown.
     * <p><p>
     * Should be used to obtain "utility" world objects such us "informations about the agent" (that
     * is unique for the agent and as the world view should be used by only one agent...) or some
     * "world statistics object".
     *
     * @param <T>
     * @param cls Class of object to be retrieved
     * @return  the only object of given class
     */
    public <T extends IWorldObject> T getSingle(Class<T> cls);
	
	/**
	 * Returns map with objects inserted according to their id.
	 * <p><p>
	 * <b>WARNING:</b> If you will do iteration over the map, you must synchronize on it.
	 * <p><p>
	 * Note that this map contains various objects, therefore you will somehow guess the correct class of the object
	 * from its id.
	 * 
	 * @return
	 */
	public Map<WorldObjectId,IWorldObject> get();

	/**
	 * Returns a world object of the specific id (if exists inside the world view). Otherwise, null is returned.
	 * <p><p>
	 * Note that there is no way to tell the correct type of returned object - you have to cast it to a correct class
	 * yourself.
	 * 
	 * @param id objects's id
	 * @return
	 */
	public IWorldObject get(WorldObjectId id);

	/**
	 * Returns a world object of the specific id and class (if exists inside the world view). Otherwise, null is returned.
	 * <p><p>
	 * 
	 * @param id objects's id
         * @param clazz the class of the object to be returned
	 * @return the object or null if not found
         * @throws ClassCastException if the object with specified id exists, but is not of specified type
	 */
	public <T extends IWorldObject> T get(WorldObjectId id, Class<T> clazz);
        
}