package cz.cuni.amis.pogamut.emohawk.examples.chefbot.task;

import com.google.common.base.Predicate;

import cz.cuni.amis.pogamut.emohawk.agent.module.action.ActionErrorPrinter;
import cz.cuni.amis.pogamut.emohawk.examples.chefbot.EmohawkVilleChefBot;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItem;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItemReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.observationMemory.memorization.item.utensil.DishwarePlateMemorization;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IIngredient;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IIngredientReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.DishwarePlateReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.IDishwarePlate;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.IScoop;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.ITurner;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.cookware.ICookwareReplica;

/** Serve an ingredient from a cookware in inventory on the plate at the specified location
 * 
 * @author Paletz
 */
public class ServeIngredientTask extends AbstractTask<ServeIngredientTask.Stage> {

	protected double searchDuration;
	protected int dishwarePlateGameObjectId;
	protected int ingredientGameObjectId;
		
	protected GoToTask goToTask;
	protected CollectTask collectScoopTask;
	protected CollectTask collectTurnerTask;
	
	/** Constructor  
	 * 
	 * @param bot bot picking up the item
	 * @param dishwarePlate dishware plate to serve on
	 * @param ingredient ingredient to serve
	 * @param searchDuration how old item memorization is acceptable
	 */
	public ServeIngredientTask( EmohawkVilleChefBot<?> bot, IDishwarePlate dishwarePlate, IIngredient ingredient, double searchDuration ) {
		super( bot, Stage.DONE, Stage.FAILED );
		this.searchDuration = searchDuration;
		this.dishwarePlateGameObjectId = dishwarePlate.getGameObjectId();
		this.ingredientGameObjectId = ingredient.getGameObjectId();
	}
	
	@Override
	protected void updateStage() {
		if ( isFinalStage( stage ) ) {
			return;
		}
		
		if ( findIngredientInInventory() == null ) {
			stage = Stage.DONE;
			return;
		}
		
		{
			boolean hasScoop = false;
			boolean hasTurner = false;
			for ( IItem item : bot.getPawn().readInventory() ) {
				if ( scoopPredicate.apply( item ) ) {
					hasScoop = true;
				}
				if ( turnerPredicate.apply( item ) ) {
					hasTurner = true;
				}
			}
			
			if ( !hasScoop ) {
				stage = Stage.COLLECT_SCOOP;
				return;
			}
			if ( !hasTurner) {
				stage = Stage.COLLECT_TURNER;
				return;
			}
		}
		
		DishwarePlateMemorization plate = findPlateInMemory();
		if ( plate == null || bot.getInfo().getTime() - plate.getMemorizationEpochTime() > searchDuration ) {
			stage = Stage.FAILED;
			return;
		}
		
		bot.getObservationMemory().setImportance( plate, 16.0 );
		
		if ( bot.getInfo().getDistance( plate.getActorLocation() ) < bot.getActionRegistry().getTakeItemFromContainerAction().getRange()*0.9 ) {
			if ( plate.getMemorizationEpochTime() == bot.getObservationMemory().getEpochTime() ) {
				stage = Stage.SERVE;
			} else {
				stage = Stage.SEARCH;
			}
		} else {
			stage = Stage.MOVE;
		}
		return;
	}
	
	@Override
	protected void stageLogic() {
		switch ( stage ) {
		case COLLECT_SCOOP:
			clearIncorrectSubtask( collectScoopTask );
			
			if ( subTask == null ) {
				subTask = collectScoopTask = new CollectTask( bot, scoopPredicate );
			}
			
			subTask.logic();
			break;
		case COLLECT_TURNER:
			clearIncorrectSubtask( collectTurnerTask );
			
			if ( subTask == null ) {
				subTask = collectTurnerTask = new CollectTask( bot, turnerPredicate );
			}
			
			subTask.logic();
			break;
		case MOVE:
			clearIncorrectSubtask( goToTask );

			if ( subTask == null ) {
				subTask = goToTask = new GoToTask( 
					bot,
					findPlateInMemory().getActorLocation(),
					bot.getActionRegistry().getStoreItemInContainerAction().getRange()*0.9 
				);
			}
			
			subTask.logic();
			break;
		case SEARCH:
			clearSubTask();
			bot.getMove().turnTo( findPlateInMemory().getActorLocation() );
			break;
		case SERVE:
			clearSubTask();
			DishwarePlateMemorization plateMemorization = findPlateInMemory();
			DishwarePlateReplica plateReplica = bot.getObservationMemory().getPreimage( plateMemorization );
			IIngredientReplica ingredientReplica = findIngredientInInventory();
			bot.getActionRegistry().getServeFoodAction().request(
				bot.getPawn(),
				ingredientReplica,
				plateReplica,
				new ActionErrorPrinter( "Serve "+ingredientReplica.getDisplayName() )
			);
			break;
		case DONE:
		case FAILED:
			clearSubTask();
			break;
		default:
			throw new AssertionError( "Unexpected stage: "+stage );
		}
	}
	
	@Override
	protected void clearSubTask() {
		super.clearSubTask();
		goToTask = null;
		collectScoopTask = null;
		collectTurnerTask = null;
	}
	
	@Override
	public String getName() {
		return "Serve";
	}
	
	protected IIngredientReplica findIngredientInInventory() {
		for ( IItemReplica item : bot.getPawn().readInventory() ) {
			if ( item instanceof ICookwareReplica ) {
				ICookwareReplica cookware = (ICookwareReplica) item;
				for ( IIngredientReplica ingredient : cookware.readIngredients() ) {
					if ( ingredient.getGameObjectId() == ingredientGameObjectId ) {
						return ingredient;
					}
				}
			}
		}
		return null;
	}
	
	protected DishwarePlateMemorization findPlateInMemory() {
		for ( DishwarePlateMemorization plate : bot.getObservationMemory().getAllByMemorization( DishwarePlateMemorization.class ) ) {
			if ( plate.getGameObjectId() == dishwarePlateGameObjectId ) {
				return plate;
			}
		}
		return null;
	}
	
	protected final Predicate<IItem> scoopPredicate = new Predicate<IItem>() {
		@Override
		public boolean apply( IItem item ) {
			return item instanceof IScoop;
		}
	};
	
	protected final Predicate<IItem> turnerPredicate = new Predicate<IItem>() {
		@Override
		public boolean apply( IItem item ) {
			return item instanceof ITurner;
		}
	};
	
	protected enum Stage {
		COLLECT_SCOOP,
		COLLECT_TURNER,
		MOVE,
		SEARCH,
		SERVE,
		DONE,
		FAILED
	}
}