package cz.cuni.amis.pogamut.emohawk.examples.chefbot.task;

import cz.cuni.amis.pogamut.emohawk.agent.module.action.ActionErrorPrinter;
import cz.cuni.amis.pogamut.emohawk.examples.chefbot.EmohawkVilleChefBot;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.observationMemory.memorization.action.trade.IBartererMemorization;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.observationMemory.memorization.item.IItemMemorization;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.action.trade.ITradeProcess;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.action.trade.TradeProcessReplica;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItem;

/** Demand an item from a pawn task
 * 
 * @author Paletz
 */
public class DemandTask extends AbstractTask<DemandTask.Stage> {

	public static final double ASK_FOR_ITEM_PERIOD = 10.0;
	
	protected double searchDuration;
	protected int itemGameObjectId;
	protected double lastAskForItemTime;
	
	protected GoToTask goToTask;

	/** Create a pick up an item task  
	 *  
	 * @param bot bot picking up the item
	 * @param item item to pick up
	 * @param searchDuration how old item memorization is acceptable
	 */
	public DemandTask(EmohawkVilleChefBot<?> bot, IItem item, double searchDuration ) {
		super(bot, Stage.DONE, Stage.FAILED);
		this.searchDuration = searchDuration;
		this.itemGameObjectId = item.getGameObjectId();
		lastAskForItemTime = - ASK_FOR_ITEM_PERIOD;
	}


	@Override
	protected void updateStage() {
		if ( isFinalStage( stage ) ) {
			return;
		}
		
		if ( isPossessed() ) {
			stage = Stage.DONE;
			return;
		}
		
		IItemMemorization<?> item = findItemInMemory();
		if ( item == null || bot.getInfo().getTime()-item.getMemorizationEpochTime() > searchDuration ) {
			stage = Stage.FAILED;
			return;
		}
		
		IBartererMemorization<?> barterer = findBartererInMemory();
		bot.getObservationMemory().setImportance( barterer, 16.0 );
		
		if ( bot.getInfo().getDistance( barterer.getActorLocation() ) < bot.getActionRegistry().getTradeAction().getRange()*0.9 ) {
			if ( item.getMemorizationEpochTime() == bot.getObservationMemory().getEpochTime() ) {
				if ( bot.getPawn().getProcess() != null ) {
					
					ITradeProcess tradeProcess = null;
					if ( bot.getPawn().getProcess() instanceof ITradeProcess ) {
						tradeProcess = (ITradeProcess) bot.getPawn().getProcess(); 
					} else {
						stage = Stage.OPEN_TRADE;
						return;
					}
					
					if ( tradeProcess != null
						&&
						tradeProcess.getBarterPartner().getGameObjectId() == barterer.getGameObjectId()
					) {
						if ( tradeProcess.getBarterPartnerProcess() != null ) {
							for ( IItem offeredItem : tradeProcess.getBarterPartnerProcess().readOffer() ) {
								if ( offeredItem.getGameObjectId() == itemGameObjectId ) {
									stage = Stage.ACCEPT_TRADE;
									return;
								}
							}
						}
						stage = Stage.ASK_FOR_ITEM;
					} else {
						stage = Stage.INTERRUPT_ACTION;
					}
				} else {
					stage = Stage.OPEN_TRADE;
				}
			} else {
				stage = Stage.SEARCH;
			}
		} else {
			stage = Stage.MOVE;
		}
	}
	
	@Override
	protected void stageLogic() {
		switch ( stage ) {
		case MOVE:
			clearIncorrectSubtask( goToTask );
			
			if ( subTask == null ) {
				subTask = goToTask = new GoToTask( bot, findBartererInMemory().getActorLocation(), bot.getActionRegistry().getPickUpItemAction().getRange()*0.9 );
			}

			subTask.logic();
			break;
		case SEARCH:
			clearSubTask();
			bot.getMove().turnTo( findBartererInMemory().getActorLocation() );
			break;
		case INTERRUPT_ACTION:
			clearSubTask();
			
			bot.getActionRegistry().getInterruptProcessAction().request(
				bot.getPawn(),
				new ActionErrorPrinter( "Interrupt action not matching intended trade" )
			);
			break;
		case OPEN_TRADE:
			clearSubTask();
			
			bot.getActionRegistry().getTradeAction().request(
				bot.getPawn(),
				bot.getObservationMemory().getPreimage( findBartererInMemory() ),
				new ActionErrorPrinter( "" )
			);
			break;
		case ASK_FOR_ITEM:
			clearSubTask();
			
			IItemMemorization<?> wantedItem = findItemInMemory();
			int itemIndex = 0;
			for ( IItemMemorization<?> item : findBartererInMemory().readInventory() ) {
				++itemIndex;
				if ( wantedItem == item ) {
					break;
				}
			}
			
			if ( lastAskForItemTime + ASK_FOR_ITEM_PERIOD < bot.getInfo().getTime() ) {
				lastAskForItemTime = bot.getInfo().getTime();
				bot.getComm().sendGlobalTextMessage( "Give me the "+findItemInMemory().getDisplayName()+", it is the item number "+itemIndex+" in your inventory!" );
			}
			break;
		case ACCEPT_TRADE:
			clearSubTask();
			((TradeProcessReplica) bot.getPawn().getProcess()).sendHasAccepted( true );
			break;
		case DONE:
		case FAILED:
			clearSubTask();
			break;
		default:
			throw new AssertionError( "Unexpected stage: "+stage );
		}
	}
	
	@Override
	protected void clearSubTask() {
		super.clearSubTask();
		goToTask = null;
	}
	
	protected boolean isPossessed() {
		for ( IItem item : bot.getPawn().readInventory() ) {
			if ( item.getGameObjectId() == itemGameObjectId ) {
				return true;
			}
		}
		
		return false;
	}
	
	@Override
	public String getName() {
		return "Demand";
	}
	
	protected IBartererMemorization<?> findBartererInMemory() {
		return (IBartererMemorization<?>) findItemInMemory().getOwnerPossessor();
	}
	
	protected IItemMemorization<?> findItemInMemory() {
		for ( IBartererMemorization<?> barterer : bot.getObservationMemory().getAllByMemorization( IBartererMemorization.class ) ) {
			for ( IItemMemorization<?> item : barterer.readInventory() ) {
				if ( item.getGameObjectId() == itemGameObjectId ) {
					return item;
				}
			}
		}
		return null;
	}
	
	protected enum Stage {
		MOVE,
		SEARCH,
		FAILED,
		INTERRUPT_ACTION,
		OPEN_TRADE,
		ASK_FOR_ITEM,
		ACCEPT_TRADE,
		DONE;
	}
}