package cz.cuni.amis.pogamut.emohawk.examples.chefbot.task;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.google.common.base.Predicate;

import cz.cuni.amis.pogamut.emohawk.examples.chefbot.EmohawkVilleChefBot;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItem;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItemReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.observationMemory.memorization.item.utensil.DishwarePlateMemorization;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IBoilableIngredient;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IFriableIngredient;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IIngredient;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IIngredientReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.cookware.ICookwareReplica;

public class CreateServedFoodTask extends AbstractTask<CreateServedFoodTask.Stage> {
	
	public static enum IngredientState {
		PLAIN,
		BOILED,
		FRIED;
	}
	
	protected List<IngredientOrder> orders;
	
	
	protected ITask acquireOrderIngredientTask;
	protected ITask serveIngredientTask;
	protected IngredientOrder currentTaskOrder;
	protected DishwarePlateMemorization servedPlate; 
	
	/** Constructor
	 * 
	 * @param bot bot
	 */
	public CreateServedFoodTask( EmohawkVilleChefBot<?> bot ) {
		super( bot, Stage.DONE, null );
		
		orders = new ArrayList<IngredientOrder>();
	}
	
	@Override
	public String getName() {
		return "Create served food";
	}
	
	public CreateServedFoodTask orderIngredient( Class<? extends IIngredient> ingredientClass, IngredientState state, float volume ) {
		IngredientOrder order = new IngredientOrder();
		order.ingredientClass = ingredientClass;
		order.state = state;
		order.volume = volume;
		orders.add( order );
		return this;
	}
	
	@Override
	protected void updateStage() {
		if ( isFinalStage( stage ) ) {
			return;
		}
				
		DishwarePlateMemorization dishwarePlate = findDishwarePlateInMemory();
		if ( 
			dishwarePlate != null
			&&
			computeProgressRating( computeProgress( dishwarePlate.readIngredients() ) )  >= orders.size()
		) {
			stage = Stage.DONE;
			return;
		}
		
		IngredientOrder orderToServe = findOrderToServe();
		
		IItem ingredient = findIngredientInInventory( orderToServe );
		if ( ingredient != null ) {
			stage = Stage.SERVE_INGREDIENT;
		} else {
			stage = Stage.CREATE_INGREDIENT;
		}
		return;
	}
	
	@Override
	protected void stageLogic() {
		switch ( stage ) {
		case CREATE_INGREDIENT:
			clearIncorrectSubtask( acquireOrderIngredientTask, currentTaskOrder != findOrderToServe() );
			
			if ( subTask == null ) {
				currentTaskOrder = findOrderToServe();
				
				switch ( currentTaskOrder.state ) {
				case BOILED:
					acquireOrderIngredientTask = new AcquireBoiledIngredientTask(
						bot,
						currentTaskOrder.getBoilableIngredientClass(),
						currentTaskOrder.volume
					);
					break;
				case FRIED:
					acquireOrderIngredientTask = new AcquireFriedIngredientTask(
						bot,
						currentTaskOrder.getFriableIngredientClass(),
						currentTaskOrder.volume
					);
					break;
				case PLAIN:
					PieceItemType pieceItemType = PieceItemType.classToType( currentTaskOrder.ingredientClass );
					if ( pieceItemType != null ) {
						acquireOrderIngredientTask = new AcquirePieceItem( bot, pieceItemType, 1 );
					} else {
						acquireOrderIngredientTask = new CollectTask( bot, currentTaskOrder );
					}
					break;
				default:
					throw new AssertionError( "Unexpected state." );
				}
				subTask = acquireOrderIngredientTask;
			}
			
			subTask.logic();
			break;
		case SERVE_INGREDIENT:
			
			IIngredientReplica ingredientToServe = findIngredientInInventory( findOrderToServe() );
			boolean shouldBeStoreTask = ingredientToServe.getOwnerPossessor() == bot.getPawn();
			boolean shouldBeServeTask = !shouldBeStoreTask;
			
			clearIncorrectSubtask( 
				serveIngredientTask, 
				currentTaskOrder != findOrderToServe() 
				|| 
				servedPlate == null
				||
				findDishwarePlateInMemory().getGameObjectId() != servedPlate.getGameObjectId()
				||
				( shouldBeStoreTask && !( serveIngredientTask instanceof StoreInContainerTask ) )
				||
				( shouldBeServeTask && !( serveIngredientTask instanceof ServeIngredientTask ) )
			);
			
			if ( subTask == null ) {
				
				servedPlate = findDishwarePlateInMemory();
				currentTaskOrder = findOrderToServe();
				
				if ( shouldBeStoreTask ) {
					serveIngredientTask = new StoreInContainerTask(
						bot, 
						servedPlate,
						ingredientToServe,
						Double.POSITIVE_INFINITY 
					);
				} else {
					serveIngredientTask = new ServeIngredientTask(
						bot, 
						servedPlate, 
						ingredientToServe,
						Double.POSITIVE_INFINITY 
					);
				}
				subTask = serveIngredientTask;
			}
			
			subTask.logic();
			break;
		case DONE:
			clearSubTask();
			break;
		default:
			throw new AssertionError( "Unexpected stage "+stage+"." );
		}
	}
	
	@Override
	protected void clearSubTask() {
		super.clearSubTask();
		acquireOrderIngredientTask = null;
		serveIngredientTask = null;
	}
	
	protected IngredientOrder findOrderToServe() {
		DishwarePlateMemorization dishwarePlate = findDishwarePlateInMemory();
		if ( dishwarePlate != null ) {
			Map<IngredientOrder,Float> progress = computeProgress( dishwarePlate.readIngredients() );
			for ( IngredientOrder order : orders ) {
				if (  order.volume > progress.get( order ) ) {
					return order;
				}
			}
		}
		
		return orders.get( 0 );
	}
	
	
	/** Find ingredient in inventory
	 * 
	 * Searches inventory and cookware in inventory.
	 * 
	 * @param order ingredient order, volume is ignored
	 * @return ingredient or null if none such exists
	 */
	protected IIngredientReplica findIngredientInInventory( IngredientOrder order ) {
		for ( IItemReplica item : bot.getPawn().readInventory() ) {
			if ( item instanceof ICookwareReplica ) {
				ICookwareReplica cookware = (ICookwareReplica) item;
				for ( IIngredientReplica ingredient : cookware.readIngredients() ) {
					if ( order.apply( ingredient ) ) {
						return ingredient;
					}
				}
			}
			
			if ( item instanceof IIngredient ) {
				IIngredientReplica ingredient = (IIngredientReplica) item;
				if ( order.apply( ingredient ) ) {
					return ingredient;
				}
			}
		}
		return null;
	}
	
	protected DishwarePlateMemorization findDishwarePlateInMemory() {
		DishwarePlateMemorization bestPlate = null;
		double bestPlateMemorizationTime = Float.NEGATIVE_INFINITY;
		double bestPlateProgressRating = 0;
		for ( DishwarePlateMemorization plate : bot.getObservationMemory().getAllByMemorization( DishwarePlateMemorization.class ) ) {
			if ( !bot.getKitchen().isWithin( plate.getActorLocation() ) ) {
				continue;
			}
			
			float progressRating = computeProgressRating( computeProgress( plate.readIngredients() ) );
			
			if ( 
				progressRating > bestPlateProgressRating 
				||
				( progressRating == bestPlateProgressRating && plate.getMemorizationEpochTime() > bestPlateMemorizationTime )
			) {
				bestPlate = plate;
				bestPlateMemorizationTime = plate.getMemorizationEpochTime();
				bestPlateProgressRating = progressRating;
			}
		}
		
		return bestPlate;
	}
	
	protected float computeProgressRating( Map<IngredientOrder,Float> plateProgressMap ) {
		float progressRating = 0.0f;
		for ( Entry<IngredientOrder,Float> entry : plateProgressMap.entrySet() ) {
			progressRating += Math.min( 1.0f, entry.getValue()/entry.getKey().volume );
		}
		return progressRating;
	}
	
	protected Map<IngredientOrder,Float> computeProgress( Collection<? extends IIngredient> ingredients ) {
		Map<IngredientOrder,Float> progressMap = new HashMap<IngredientOrder,Float>();
		for ( IngredientOrder order : orders ) {
			progressMap.put( order, 0f );
		}
		
		for ( IIngredient ingredient : ingredients ) {
			for ( IngredientOrder order : orders ) { 
				if ( 
					order.apply( ingredient )
					&&
					progressMap.get( order ) < 1.0
				) {
					progressMap.put( order, progressMap.get( order ) + ingredient.getVolume() );
					break;
				}
			}
		}
		
		return progressMap;
	}
			
	protected enum Stage {
		CREATE_INGREDIENT,
		SERVE_INGREDIENT,
		DONE
	}
	
	protected class IngredientOrder implements Predicate<IItem> {
		public Class<? extends IIngredient> ingredientClass;
		public IngredientState state;
		public float volume;
		
		@Override
		public boolean apply( IItem item ) {
			if ( !ingredientClass.isInstance( item ) ) {
				return false;
			}
			
			switch ( state ) {
			case BOILED:
				return ((IBoilableIngredient) item).getBoiledness() == 1.0;
			case FRIED:
				return (
					((IFriableIngredient) item).getBottomFriedness() == 1.0
					&&
					((IFriableIngredient) item).getTopFriedness() == 1.0
				);
			case PLAIN:
				return true;
			default:
				throw new AssertionError( "Unexpected state." );
			}
		}
		
		public Class<? extends IBoilableIngredient> getBoilableIngredientClass() {
			assert( state == IngredientState.BOILED );
			@SuppressWarnings("unchecked")
			Class<? extends IBoilableIngredient> retval = (Class<? extends IBoilableIngredient>) ingredientClass;
			return retval;
		}
		
		public Class<? extends IFriableIngredient> getFriableIngredientClass() {
			assert( state == IngredientState.FRIED );
			@SuppressWarnings("unchecked")
			Class<? extends IFriableIngredient> retval = (Class<? extends IFriableIngredient>) ingredientClass;
			return retval;
		}
	}
}
