package cz.cuni.amis.pogamut.emohawk.examples.chefbot.task;

import com.google.common.base.Predicate;

import cz.cuni.amis.pogamut.emohawk.agent.module.action.ActionErrorPrinter;
import cz.cuni.amis.pogamut.emohawk.examples.chefbot.EmohawkVilleChefBot;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItem;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.observationMemory.memorization.StovePlateMemorization;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.observationMemory.memorization.item.utensil.cookware.ICookwareMemorization;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.IStovePlate;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.StovePlateReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.StoveReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IIngredient;

/** A task to cook the contents of the cookware on the specified stove plate 
 * 
 * @author Paletz
 */
public abstract class AbstractCookTask extends AbstractTask<AbstractCookTask.Stage> {

	protected double searchDuration;
	protected int stovePlateGameObjectId;
	
	protected GoToTask goToTask;
	protected CollectTask collectToolTask;
	
	/** Constructor
	 * 
	 * @param bot bot
	 * @param stovePlate stove plate
	 * @param searchDuration how old board memorization is acceptable
	 */
	public AbstractCookTask( EmohawkVilleChefBot<?> bot, IStovePlate stovePlate, double searchDuration ) {
		super( bot, Stage.DONE, Stage.FAILED );
		this.stovePlateGameObjectId = stovePlate.getGameObjectId();
		this.searchDuration = searchDuration;
	}
	
	protected abstract Predicate<IItem> getToolPredicate();
	protected abstract boolean isCooked( IIngredient ingredient );
	protected abstract float getCookingRange();
	
	protected abstract float getMinCookingTemperature();
	protected abstract float getMaxCookingTemperature();
	protected abstract void cookingLogic();
	protected abstract int getTargetStovePower();
	
	@Override
	protected void updateStage() {
		if ( isFinalStage( stage ) ) {
			return;
		}
		
		StovePlateMemorization stovePlate = findStovePlateInMemory();
		ICookwareMemorization<?> cookware = findCookwareInMemory();
		if ( 
			stovePlate == null 
			||
			cookware == null
			||
			bot.getInfo().getTime()-stovePlate.getMemorizationEpochTime() > searchDuration
		) {
			stage = Stage.FAILED;
			return;
		}
		
		{
			boolean allCooked = true;
			for ( IIngredient ingredient : cookware.readIngredients() ) {
				if ( !isCooked(ingredient) ) {
					allCooked = false;
				}
			}
			if ( allCooked ) {
				stage = Stage.DONE;
				return;
			}
		}
		
		{
			boolean hasTool = false;
			for ( IItem item : bot.getPawn().readInventory() ) {
				if ( getToolPredicate().apply( item ) ) {
					hasTool = true;
					break;
				}
			}
			
			if ( !hasTool ) {
				stage = Stage.COLLECT_TOOL;
				return;
			}
		}
				
		if ( bot.getInfo().getDistance( stovePlate.getActorLocation() ) < getCookingRange()*0.5 ) {
			if ( stovePlate.getMemorizationEpochTime() == bot.getObservationMemory().getEpochTime() ) {
				if ( cookware.getTemperature() < getMinCookingTemperature() || getMaxCookingTemperature() < cookware.getTemperature() ) {
					if ( stovePlate.getStove() != null ) {
						stage = Stage.SET_POWER;
					} else {
						stage = Stage.SEARCH;
					}
					return;
				}
				stage = Stage.COOK;
			} else {
				stage = Stage.SEARCH;
			}
			return;
		} else {
			stage = Stage.MOVE;
			return;
		}
		
	}
	
	@Override
	protected void stageLogic() {
		switch ( stage ) {
		case COLLECT_TOOL:
			clearIncorrectSubtask( collectToolTask );
			
			if ( subTask == null ) {
				subTask = collectToolTask = new CollectTask( bot, getToolPredicate() ); 
			}
			subTask.logic();
			break;
		case MOVE:
			clearIncorrectSubtask( goToTask );
			
			if ( subTask == null ) {
				subTask = goToTask = new GoToTask( bot, findStovePlateInMemory().getActorLocation(), getCookingRange()*0.5 );
			}
			
			subTask.logic();
			break;
		case SEARCH:
			clearSubTask();
			bot.getMove().turnTo( findStovePlateInMemory().getActorLocation() );
			break;
		case SET_POWER:
			clearSubTask();
			
			StovePlateReplica stovePlate = bot.getObservationMemory().getPreimage( findStovePlateInMemory() );
			StoveReplica stove = stovePlate.getStove();
			for ( int i=0; i<StoveReplica.PLATE_COUNT; ++i ) {
				if ( stove.getPlate( i ) == stovePlate ) {
					if ( stove.getPower( i ) != getTargetStovePower() ) {
						bot.getActionRegistry().getSetStovePowerAction().request(
							bot.getPawn(),
							stove,
							i,
							getTargetStovePower(),
							new ActionErrorPrinter( "Set stove power")
						);
					}
				}
			}
			break;
		case COOK:
			clearSubTask();
			
			cookingLogic();
			break;
		case DONE:
		case FAILED:
			clearSubTask();
			break;
		}
	}
	
	@Override
	protected void clearSubTask() {
		super.clearSubTask();
		goToTask = null;
		collectToolTask = null;
	}
	
	protected ICookwareMemorization<?> findCookwareInMemory() {
		StovePlateMemorization stovePlate = findStovePlateInMemory();
		if ( stovePlate == null || stovePlate.readInventory().isEmpty() ) {
			return null;
		}
		return (ICookwareMemorization<?>) stovePlate.readInventory().iterator().next();
	}
	
	protected StovePlateMemorization findStovePlateInMemory() {
		for ( StovePlateMemorization stovePlate : bot.getObservationMemory().getAllByMemorization( StovePlateMemorization.class ) ) {
			if ( stovePlate.getGameObjectId() == stovePlateGameObjectId ) {
				return stovePlate;
			}
		}
		return null;
	}
	
	protected enum Stage {
		COLLECT_TOOL,
		MOVE,
		SEARCH,
		SET_POWER,
		COOK,
		DONE,
		FAILED
	}
}
