package cz.cuni.amis.pogamut.emohawk.examples.chefbot.task;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Predicate;
import com.google.common.base.Suppliers;

import cz.cuni.amis.pogamut.emohawk.agent.module.action.ActionErrorPrinter;
import cz.cuni.amis.pogamut.emohawk.examples.chefbot.EmohawkVilleChefBot;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.observationMemory.memorization.item.IItemMemorization;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItem;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItemReplica;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.ISubstanceItem;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.ISubstanceItemReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.observationMemory.memorization.StovePlateMemorization;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.observationMemory.memorization.item.utensil.cookware.ICookwareMemorization;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.ingredient.IIngredient;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.cookware.ICookware;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.cookware.ICookwareReplica;

/** Acquired cooked ingredient task. 
 * 
 * The result is a cooked ingredient in a cookware.
 * 
 * @author Paletz
 */
public abstract class AbstractAcquireCookedIngredientTask extends AbstractTask<AbstractAcquireCookedIngredientTask.Stage> {

	public static final double MAX_GOAL_VOLUME = 1.0;
	
	protected Class<? extends IIngredient> ingredientClass;
	protected double goalVolume;
	
	protected CollectTask collectCookwareTask = null;
	protected SearchTask searchTask = null;
	protected StoreInContainerTask placeCookwareTask = null;
	protected CollectTask collectCookingLiquidTask = null;
	protected StoreInContainerTask addCookingLiquidTask = null;
	protected ITask acquireIngredientTask = null;
	protected StoreInContainerTask addIngredientTask = null; 
	protected CollectTask collectToolTask = null;
	protected CollectTask collectFinishedTask = null;
	protected ITask cookTask = null;
	
	/** Constructor
	 * 
	 * @param bot bot
	 * @param ingredientClass ingredient class (should be general interface, not memorization or replica)
	 * @param goalVolume goal cooked ingredient volume, supported maximum {@link #MAX_GOAL_VOLUME}
	 */
	public AbstractAcquireCookedIngredientTask( EmohawkVilleChefBot<?> bot, Class<? extends IIngredient> ingredientClass, double goalVolume ) {
		super(bot, Stage.DONE, null );
		this.ingredientClass = ingredientClass;
		this.goalVolume = goalVolume;
		assert( goalVolume <= MAX_GOAL_VOLUME );
	}
	

	public Class<? extends IIngredient> getIngredientClass() {
		return ingredientClass;
	}
	
	/** Tell if the cookware is compatible with the task 
	 * 
	 * @param cookware cookware
	 * @return true if the cookware is compatible with the task
	 */
	protected abstract boolean isCompatibleCookware( ICookware cookware );
	
	protected abstract Predicate<IItem> getCookingLiquidPredicate();
	protected abstract float getMinCookingLiquidVolume();
	protected abstract float getMaxCookingLiquidVolume();
	
	protected abstract boolean isCooked( IIngredient ingredient );
	
	protected abstract ITask makeCookTask();
	
	@Override
	protected void updateStage() {
		if ( isFinalStage( stage ) ) { 
			return;
		}
		
		ICookwareMemorization<?> cookware = findCookwareInMemory( CookwareState.POSSESED_FINISHED );
		if ( cookware != null ) {
			stage = Stage.DONE;
			return;
		}
		
		cookware = findCookwareInMemory( CookwareState.FINISHED );
		if ( cookware != null ) {
			if ( cookware.getOwnerPossessor() == null ) {
				bot.getObservationMemory().setImportance( cookware, 16.0 );
			}
			
			stage = Stage.COLLECT_FINISHED;
			return;
		}
				
		cookware = findCookwareInMemory( CookwareState.PLACED_READY );
		if ( cookware != null ) {
			stage = Stage.COOK;
			return;
		}
		
		cookware = findCookwareInMemory( CookwareState.PLACED_WITH_LIQUID );
		if ( cookware != null ) {
			if ( findIngredientInInventory() != null ) {
				stage = Stage.ADD_INGREDIENT;
			} else {
				stage = Stage.ACQUIRE_INGREDIENT;
			}
			return;
		}
		
		cookware = findCookwareInMemory( CookwareState.PLACED );
		if ( cookware != null ) {
			CookwareInfo cookwareInfo = inspectCookware( cookware );
			
			ISubstanceItem cookingLiquid = findCookingLiquidInInventory( getMaxCookingLiquidVolume() - cookwareInfo.cookingLiquidVolume );
			if ( cookingLiquid != null ) {
				stage = Stage.ADD_COOKING_LIQUID;
				return;
			}
			
			cookingLiquid = findCookingLiquidInInventory( Float.POSITIVE_INFINITY );
			if ( cookingLiquid != null ) {
				stage = Stage.SPLIT_COOKING_LIQUID;
			} else {
				stage = Stage.COLLECT_COOKING_LIQUID;
			}
			
			return;
		}
		
		cookware = findCookwareInMemory( CookwareState.POSSESED );
		if ( cookware != null ) {
			StovePlateMemorization stovePlate = findFreeStovePlateInMemory();
			if ( stovePlate != null ) {
				bot.getObservationMemory().setImportance( stovePlate, 16.0 );
				stage = Stage.PLACE_COOKWARE;
			} else {
				stage = Stage.SEARCH_STOVE;
			}
			return;
		}
		
		stage = Stage.COLLECT_COOKWARE;
	}

	@Override
	protected void stageLogic() {
		switch ( stage ) {
		case COLLECT_COOKWARE:
			clearIncorrectSubtask( collectCookwareTask );
			
			if ( subTask == null ) {
				subTask = collectCookwareTask = new CollectTask( bot, freeCookwarePredicate ); 
			}
			subTask.logic();
			break;
		case SEARCH_STOVE:
			clearIncorrectSubtask( searchTask );
			
			if ( subTask == null ) {
				subTask = searchTask = new SearchTask( bot, bot.getKitchen().getNavPoints(), Suppliers.ofInstance( false ) ); 
			}
			subTask.logic();
			break;
		case PLACE_COOKWARE:
			clearIncorrectSubtask( placeCookwareTask );
			
			if ( subTask == null ) {
				StovePlateMemorization stovePlate = findFreeStovePlateInMemory();
				subTask = placeCookwareTask = new StoreInContainerTask( bot, stovePlate, findCookwareInMemory( CookwareState.POSSESED ), Double.POSITIVE_INFINITY ); 
			}
			subTask.logic();
			break;
		case COLLECT_COOKING_LIQUID:
			clearIncorrectSubtask( collectCookingLiquidTask );
			
			if ( subTask == null ) {
				subTask = collectCookingLiquidTask = new CollectTask( bot, getCookingLiquidPredicate() ); 
			}
			subTask.logic();
			break;
		case SPLIT_COOKING_LIQUID:
			clearSubTask();
			
			{
				ISubstanceItemReplica cookingLiquid = findCookingLiquidInInventory( Float.POSITIVE_INFINITY );
				ICookware cookware = findCookwareInMemory( CookwareState.PLACED );
				CookwareInfo cookwareInfo = inspectCookware( cookware );
				float splitVolume = (getMinCookingLiquidVolume()+getMaxCookingLiquidVolume())/2 - cookwareInfo.cookingLiquidVolume;
				bot.getActionRegistry().getSplitItemAction().requestSplitSubstance(
					bot.getPawn(),
					cookingLiquid,
					splitVolume,
					new ActionErrorPrinter( "Split cooking liquid")
				);
			}
			break;
		case ADD_COOKING_LIQUID:
			clearIncorrectSubtask( addCookingLiquidTask );
			
			if ( subTask == null ) {
				ICookwareMemorization<?> cookware = findCookwareInMemory( CookwareState.PLACED );
				ISubstanceItemReplica cookingLiquid = findCookingLiquidInInventory( getMaxCookingLiquidVolume() - inspectCookware( findCookwareInMemory( CookwareState.PLACED ) ).cookingLiquidVolume );
				subTask = addCookingLiquidTask = new StoreInContainerTask(
					bot, 
					cookware,
					cookingLiquid,
					Double.POSITIVE_INFINITY 
				); 
			}
			subTask.logic();
			break;
		case ACQUIRE_INGREDIENT:
			clearIncorrectSubtask( acquireIngredientTask );
			
			if ( subTask == null ) {
				PieceItemType pieceItemType = PieceItemType.classToType( ingredientClass );
				if ( pieceItemType != null ) {
					acquireIngredientTask = new AcquirePieceItem( bot, pieceItemType, 1 );
				} else {
					acquireIngredientTask = new CollectTask( bot, ingredientPredicate );
				}
				subTask = acquireIngredientTask; 
			}
			subTask.logic();
			break;
		case ADD_INGREDIENT:
			clearIncorrectSubtask( addIngredientTask );
			
			if ( subTask == null ) {	
				ICookwareMemorization<?> cookware = findCookwareInMemory( CookwareState.PLACED_WITH_LIQUID );
				subTask = addIngredientTask = new StoreInContainerTask(
					bot, 
					cookware,
					findIngredientInInventory(),
					Double.POSITIVE_INFINITY 
				);
			}
			subTask.logic();
			break;
		case COOK:
			clearIncorrectSubtask( cookTask );
			
			if ( subTask == null ) {
				subTask = cookTask = makeCookTask(); 
			}
			subTask.logic();
			break;
		case COLLECT_FINISHED:
			clearIncorrectSubtask( collectFinishedTask );
			
			if ( subTask == null ) {
				subTask = collectFinishedTask = new CollectTask( 
					bot, 
					new Predicate<IItem>() {
						@Override
						public boolean apply( IItem item ) {
							if ( item instanceof ICookware ) {
								ICookware cookware = (ICookware) item;
								return (
									isCompatibleCookware( cookware )
									&&
									checkCookwareInternalState(cookware, CookwareState.FINISHED ) 
								);
							} else {
								return false;
							}
						}
					}
				); 
			}
			subTask.logic();
			break;
		case DONE:
			clearSubTask();
			break;
		}
	}
	
	@Override
	protected void clearSubTask() {
		super.clearSubTask();
		collectCookwareTask = null;
		searchTask = null;
		placeCookwareTask = null;
		collectCookingLiquidTask = null;
		addCookingLiquidTask = null;
		acquireIngredientTask = null;
		addIngredientTask = null;
		collectToolTask = null;
		collectFinishedTask = null;
		cookTask = null;
	}
	
	protected StovePlateMemorization findFreeStovePlateInMemory() {
		for ( StovePlateMemorization stovePlate : bot.getObservationMemory().getAllByMemorization( StovePlateMemorization.class ) ) {
			if ( stovePlate.readInventory().isEmpty() && bot.getKitchen().isWithin( stovePlate.getActorLocation() ) ) {
				return stovePlate;
			}
		}
		
		return null;
	}
	
	protected IIngredient findIngredientInInventory() {
		for ( IItem item : bot.getPawn().readInventory() ) {
			if ( ingredientClass.isInstance( item ) ) {
				return (IIngredient) item;
			}
		}
		return null;
	}
	
	protected ISubstanceItemReplica findCookingLiquidInInventory( float maxVolume ) {
		
		for ( IItemReplica item : bot.getPawn().readInventory() ) {
			if ( getCookingLiquidPredicate().apply( item ) ) {
				ISubstanceItemReplica cookingLiquid = (ISubstanceItemReplica) item;
				if ( cookingLiquid.getAmount() < maxVolume ) {
					return cookingLiquid;
				}
			}
		}
		
		return null;
	}
	
	protected ICookwareMemorization<?> findCookwareInMemory( CookwareState requiredState ) {

		List<ICookwareMemorization<?>> candidates = new ArrayList<ICookwareMemorization<?>>();
		
		switch ( requiredState ) {
		case FREE:
			for ( ICookwareMemorization<?> cookware : bot.getObservationMemory().getAllByMemorization( ICookwareMemorization.class ) ) {
				if ( bot.getKitchen().isWithin( cookware.getActorLocation() ) ) {
					candidates.add( cookware );
				}
			}
			break;
		case POSSESED:
		case POSSESED_FINISHED:
			for ( IItemReplica item : bot.getPawn().readInventory() ) {
				if ( item instanceof ICookwareReplica ) {
					candidates.add( (ICookwareMemorization<?>) bot.getObservationMemory().getMemorization( item ) );
				}
			}
			break;
		case PLACED:
		case PLACED_WITH_LIQUID:
		case PLACED_READY:
		case FINISHED:
			for ( StovePlateMemorization stovePlate : bot.getObservationMemory().getAllByMemorization( StovePlateMemorization.class ) ) {
				if ( !bot.getKitchen().isWithin( stovePlate.getActorLocation() ) ) {
					continue;
				}
				for ( IItemMemorization<?> item : stovePlate.readInventory() ) {
					if ( item instanceof ICookwareMemorization<?> ) {
						candidates.add( (ICookwareMemorization<?>) item );
					}
				}
			}
			break;
		}
		
		for ( ICookwareMemorization<?> cookware : candidates ) {
			if ( isCompatibleCookware( cookware )
				&&
				checkCookwareInternalState(cookware, requiredState ) 
			) {
				return cookware;
			}
		}
		return null;
	}
	
	protected boolean checkCookwareInternalState( ICookware cookware, CookwareState requiredState ) {
		
		CookwareInfo cookwareInfo = inspectCookware( cookware );
		
		if ( 
			cookwareInfo.hasContaminants 
			||
			getMaxCookingLiquidVolume() < cookwareInfo.cookingLiquidVolume 
		) {
			return false;
		}
		
		switch ( requiredState ) {
		case FINISHED:
		case POSSESED_FINISHED:
			return cookwareInfo.cookedIngredientVolume >= goalVolume;
		case PLACED_READY:
			return (
				getMinCookingLiquidVolume() <= cookwareInfo.cookingLiquidVolume
				&&
				goalVolume < cookwareInfo.ingredientVolume
			);
		case PLACED_WITH_LIQUID:
			return (
				getMinCookingLiquidVolume() <= cookwareInfo.cookingLiquidVolume
			);
		case POSSESED:
		case FREE:
		case PLACED:
			return true;
		default:
			throw new AssertionError( "Unexpected state "+requiredState+"." );
		}	
	}
	
	protected class CookwareInfo {
		public float cookingLiquidVolume = 0;
		public float ingredientVolume = 0;
		public float cookedIngredientVolume = 0;
		public boolean hasContaminants = false;
	}
	
	protected CookwareInfo inspectCookware( ICookware cookware ) {
		CookwareInfo retval = new CookwareInfo();
		
		for ( IIngredient ingredient : cookware.readIngredients() ) {
			if ( getCookingLiquidPredicate().apply( ingredient ) ) {
				retval.cookingLiquidVolume += ingredient.getVolume();
				continue;
			}
			
			if ( ingredientClass.isInstance( ingredient ) ) {
				retval.ingredientVolume += ingredient.getVolume();
				if ( isCooked( ingredient ) ) {
					retval.cookedIngredientVolume += ingredient.getVolume();
				}
				continue;
			}
			
			retval.hasContaminants = true;
		}
		
		return retval;
	}
	
	protected final Predicate<IItem> ingredientPredicate = new Predicate<IItem>() {
		@Override
		public boolean apply( IItem item ) {
			return ingredientClass.isInstance( item );
		}	
	};
	
	protected final Predicate<IItem> freeCookwarePredicate = new Predicate<IItem>() {
		@Override
		public boolean apply( IItem item ) {
			if ( item instanceof ICookware ) {
				ICookware cookware = (ICookware) item;
				return (
					isCompatibleCookware( cookware )
					&&
					checkCookwareInternalState( cookware, CookwareState.FREE )
				);
			} else {
				return false;
			}
		}
	};
	
	protected enum Stage {
		COLLECT_COOKWARE,
		SEARCH_STOVE,
		PLACE_COOKWARE,
		COLLECT_COOKING_LIQUID,
		SPLIT_COOKING_LIQUID,
		ADD_COOKING_LIQUID,
		ACQUIRE_INGREDIENT,
		ADD_INGREDIENT,
		COOK,
		COLLECT_FINISHED,
		DONE
	}
	
	protected enum CookwareState {
		FREE, // cookware in the world with ingredient or space for ingredient
		POSSESED, // cookware in the inventory with ingredient or space for ingredient
		PLACED, // cookware placed on stove with enough space for ingredient
		PLACED_WITH_LIQUID,
		PLACED_READY, // cookware placed on stove with raw ingredient
		FINISHED, // cookware with cooked ingredient
		POSSESED_FINISHED // cookware with cooked ingredient in inventory
	}
}
