package cz.cuni.amis.pogamut.emohawk.examples.chefbot.task;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Predicate;
import com.google.common.base.Suppliers;
import com.google.common.collect.Collections2;

import cz.cuni.amis.pogamut.emohawk.examples.chefbot.EmohawkVilleChefBot;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.observationMemory.memorization.action.trade.IBartererMemorization;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.observationMemory.memorization.item.IItemMemorization;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.observationMemory.memorization.item.container.IContainerMemorization;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItem;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.container.IContainer;

/** Find and collect an item task
 * 
 * @author Paletz
 *
 */
public class CollectTask extends AbstractTask<CollectTask.Stage> {
	
	protected Predicate<IItem> itemPredicate;
	protected int initialPossessedMatchingItemCount;
	
	protected SearchTask searchTask;
	protected PickUpTask pickUpTask;
	protected TakeFromContainerTask takeFromContainerTask;
	protected DemandTask demandTask;
	
	/** Create a collect an item task  
	 * 
	 * @param bot bot
	 * @param itemPredicate predicate identifying the item
	 */
	public CollectTask( EmohawkVilleChefBot<?> bot, Predicate<IItem> itemPredicate ) {
		super( bot, Stage.DONE, null );
		this.itemPredicate = itemPredicate;
		this.initialPossessedMatchingItemCount = countPossessed();
	}
	
	/** Find collectable item
	 * 
	 * @param bot bot
	 * @param itemPredicate predicate identifying the item
	 * @return found item or null
	 */
	public static IItemMemorization<?> findItemInMemory( EmohawkVilleChefBot<?> bot, Predicate<IItem> itemPredicate ) {
		List<IItemMemorization<?>> candidates = new ArrayList<IItemMemorization<?>>();
		for ( IItemMemorization<?> item : bot.getObservationMemory().getAllByMemorization( IItemMemorization.class ) ) {
			if ( bot.getKitchen().isWithin( item.getActorLocation() ) && bot.getPawn().canPickUp( item ) ) {
				candidates.add( item );
			}
		}
		for ( IContainerMemorization<?> container : bot.getObservationMemory().getAllByMemorization( IContainerMemorization.class ) ) {
			if ( !bot.getKitchen().isWithin( container.getActorLocation() ) ) {
				continue;
			}
			for ( IItemMemorization<?> item : container.readInventory() ) {
				if ( container.canRelease( item ) ) {
					candidates.add( item );				
				}
			}
		}
		for ( IBartererMemorization<?> barterer : bot.getObservationMemory().getAllByMemorization( IBartererMemorization.class ) ) {
			if ( 
				barterer.getGameObjectId() == bot.getPawn().getGameObjectId() 
				||
				!bot.getKitchen().isWithin( barterer.getActorLocation() )
			) {
				continue;
			}
			for ( IItemMemorization<?> item : barterer.readInventory() ) {
				candidates.add( item );				
			} 
		}
		
		for ( IItemMemorization<?> item : candidates ) {
			if ( itemPredicate.apply( item ) ) {
				return item;
			}
		}
		
		return null;
	}
	
	@Override
	protected void updateStage() {
		if ( isFinalStage( stage ) ) {
			return;
		}
		
		if ( countPossessed() > initialPossessedMatchingItemCount ) {
			stage = Stage.DONE;
			return;
		}
		
		IItemMemorization<?> item = findItemInMemory();
		if ( item != null ) {
			if ( item.getActorLocation() != null ) {
				stage = Stage.PICK_UP;
			} else if ( item.getOwnerPossessor() instanceof IContainer ) {
				stage = Stage.TAKE;
			} else {
				stage = Stage.DEMAND;
			}
		} else {
			stage = Stage.SEARCH;
		}
	}
	
	@Override
	protected void stageLogic() {
		switch ( stage ) {
		case SEARCH:
			clearIncorrectSubtask( searchTask );
			
			if ( subTask == null ) {
				subTask = searchTask = new SearchTask( bot, bot.getKitchen().getNavPoints(), Suppliers.ofInstance( false ) ); 
			}
			
			subTask.logic();
			break;
		case PICK_UP:
			clearIncorrectSubtask( pickUpTask );
			
			if ( subTask == null ) {
				subTask = pickUpTask = new PickUpTask( bot, findItemInMemory(), Double.POSITIVE_INFINITY );
			}
			
			subTask.logic();
			break;
		case TAKE:
			clearIncorrectSubtask( takeFromContainerTask );
			
			if ( subTask == null ) {
				IItem item = findItemInMemory();
				IContainer container = (IContainer) item.getOwnerPossessor();
				subTask = takeFromContainerTask = new TakeFromContainerTask( 
					bot, 
					container,
					item,
					Double.POSITIVE_INFINITY 
				);
			}
			
			subTask.logic();
			break;
		case DEMAND:
			clearIncorrectSubtask( demandTask );
			
			if ( subTask == null ) {
				subTask = demandTask = new DemandTask( bot, findItemInMemory(), Double.POSITIVE_INFINITY );
			}
			
			subTask.logic();
			break;			
		case DONE:
			break;
		}
	}
	
	@Override
	protected void clearSubTask() {
		super.clearSubTask();
		searchTask = null;
		pickUpTask = null;
		takeFromContainerTask = null;
		demandTask = null;
	}
	
	@Override
	public String getName() {
		return "Collect";
	}
	
	protected IItemMemorization<?> findItemInMemory() {
		return findItemInMemory( bot, itemPredicate );
	}
	
	protected int countPossessed() {
		return Collections2.filter( bot.getPawn().readInventory(), itemPredicate ).size();
	}	
				
	protected enum Stage {
		SEARCH,
		PICK_UP,
		TAKE,
		DEMAND,
		DONE;
	}
}
