package cz.cuni.amis.pogamut.emohawk.examples.chefbot.task;

import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.collect.Collections2;

import cz.cuni.amis.pogamut.emohawk.agent.module.action.ActionErrorPrinter;
import cz.cuni.amis.pogamut.emohawk.examples.chefbot.EmohawkVilleChefBot;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.ICountableItem;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItem;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IItemReplica;
import cz.cuni.amis.pogamut.emohawkRpgBase.agent.module.replication.image.item.IMergeableItemReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.observationMemory.memorization.item.utensil.ChoppingBoardMemorization;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.IChoppableItemReplica;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.IChoppingBoard;
import cz.cuni.amis.pogamut.emohawkVille.agent.module.replication.image.item.utensil.IKitchenKnife;

public class AcquirePieceItem extends AbstractTask<AcquirePieceItem.Stage> {

	protected PieceItemType pieceItemType;
	protected int goalCount;
	
	protected CollectTask collectKnifeTask;
	protected CollectTask collectPrecursorTask;
	protected SearchTask findBoardTask;
	protected TakeFromContainerTask clearBoardTask;
	protected ChopTask chopTask;
	protected ITask putPrecursorOnBoardTask;
	protected CollectTask collectPieceItemTask;
	
	public AcquirePieceItem( EmohawkVilleChefBot<?> bot, PieceItemType pieceItemType, int goalCount ) {
		super( bot, Stage.DONE, null );
		this.pieceItemType = pieceItemType;
		this.goalCount = goalCount;
	}
	
	public PieceItemType getPieceItemType() {
		return pieceItemType;
	}
		
	@Override
	protected void updateStage() {
		if ( isFinalStage( stage ) ) {
			return;
		}
		
		if ( countPossessedPieceItems( CountMethod.FIRST_STACK ) >= goalCount ) {
			stage = Stage.DONE;
			return;
		}
		
		if  ( countPossessedPieceItems( CountMethod.ALL ) >= goalCount ) {
			stage = Stage.MERGE;
			return;
		}
		
		if ( CollectTask.findItemInMemory( bot, pieceItemPredicate ) != null ) {
			stage = Stage.COLLECT_PIECE_ITEM;
			return;
		}
		
		if ( 
			bot.getActionRegistry().getChopAction().findKitchenKnife( bot.getPawn().readInventory() ) == null 
			&&
			CollectTask.findItemInMemory( bot, kitchenKnifePredicate ) != null 
		) {
			stage = Stage.COLLECT_KNIFE;
			return;
		}
				
		ChoppingBoardMemorization board = findBoardInMemory( BoardState.READY );
		if ( board != null ) {
			stage = Stage.CHOP_PRECURSOR;
			return;
		}
		
		if ( findPrecursorInInventory() == null )
		{
			stage = Stage.COLLECT_PRECURSOR;
			return;
		}
		
		board = findBoardInMemory( BoardState.EMPTY );
		if ( board != null ) {
			stage = Stage.PUT_PRECURSOR_ON_BOARD;
			return;
		}
		
		board = findBoardInMemory( BoardState.BLOCKED );
		if ( board != null ) {
			stage = Stage.CLEAR_BOARD;
			return;
		}
		
		stage = Stage.FIND_BOARD;
	}
	
	@Override
	protected void stageLogic() {
		switch ( stage ) {
		case COLLECT_KNIFE:
			clearIncorrectSubtask( collectKnifeTask );
			
			if ( subTask == null ) {
				subTask = collectKnifeTask = new CollectTask( bot, kitchenKnifePredicate );
			}
			
			subTask.logic();
			break;
		case COLLECT_PRECURSOR:
			clearIncorrectSubtask( collectPrecursorTask );
						
			if ( subTask == null ) {
				subTask = collectPrecursorTask = new CollectTask( bot, precursorPredicate );
			}
			
			subTask.logic();
			break;
		case FIND_BOARD:
			clearIncorrectSubtask( findBoardTask );
			
			if ( subTask == null ) {
				subTask = findBoardTask = new SearchTask( bot, bot.getKitchen().getNavPoints(), hasFoundBoardSupplier );
			}
			
			subTask.logic();
			break;
		case CLEAR_BOARD:
			clearIncorrectSubtask( clearBoardTask );
			
			if ( subTask == null ) {
				IChoppingBoard board = findBoardInMemory( BoardState.BLOCKED );
				subTask = clearBoardTask = new TakeFromContainerTask( 
					bot,
					board,
					board.readInventory().iterator().next(),
					Double.POSITIVE_INFINITY 
				);
			}
			
			subTask.logic();
			break;
		case PUT_PRECURSOR_ON_BOARD:
			clearIncorrectSubtask( putPrecursorOnBoardTask );
			
			if ( subTask == null ) {
				subTask = putPrecursorOnBoardTask = new StoreInContainerTask( 
					bot, 
					findBoardInMemory( BoardState.EMPTY ),
					findPrecursorInInventory(),
					Double.POSITIVE_INFINITY 
				);
			}
			
			subTask.logic();
			break;
		case CHOP_PRECURSOR:
			clearIncorrectSubtask( chopTask );
			
			if ( subTask == null ) {
				subTask = chopTask = new ChopTask( bot, findBoardInMemory( BoardState.READY ), Double.POSITIVE_INFINITY );
			}
			
			subTask.logic();
			break;
		case COLLECT_PIECE_ITEM:
			clearIncorrectSubtask( collectPieceItemTask );
						
			if ( subTask == null ) {
				subTask = collectPieceItemTask = new CollectTask( bot, pieceItemPredicate );
			}
			
			subTask.logic();
			break;
			
		case MERGE:
			clearSubTask();
			
			IMergeableItemReplica [] merged = new IMergeableItemReplica[]{ null, null };
			int i=0;
			for ( IItemReplica item : bot.getPawn().readInventory() ) {
				if ( pieceItemPredicate.apply( item ) ) {
					merged[i] = (IMergeableItemReplica) item;
					i++;
					if ( i>=2 ) {
						break;
					}
				}
			}
			bot.getActionRegistry().getMergeItemAction().request(
				bot.getPawn(),
				merged[0],
				merged[1],
				new ActionErrorPrinter( "Merge piece item" )
			);
			break;
		case DONE:
			clearSubTask();
			break;
		}
	}
	
	@Override
	protected void clearSubTask() {
		super.clearSubTask();
		collectKnifeTask = null;
		collectPrecursorTask = null;
		findBoardTask = null;
		clearBoardTask = null;
		chopTask = null;
		putPrecursorOnBoardTask = null;
		collectPieceItemTask = null;
	}

	@Override
	public String getName() {
		return "AcquirePieceItem["+pieceItemType.toString()+"]";
	}
	
	protected int countPossessedPieceItems( CountMethod countMethod ) {
		int count = 0;
		for ( IItem item : bot.getPawn().readInventory() ) {
			if ( pieceItemPredicate.apply( item ) ) {
				count +=  ((ICountableItem) item).getCount();
				if ( countMethod == CountMethod.FIRST_STACK ) {
					return count;
				}
			}
		}
		return count;
	}
	
	protected IChoppableItemReplica findPrecursorInInventory() {
		for ( IItem item : bot.getPawn().readInventory() ) {
			if ( precursorPredicate.apply( item ) ) {
				return (IChoppableItemReplica) item;
			}
		}
		return null;
	}
	
	protected ChoppingBoardMemorization findBoardInMemory( BoardState requiredState ) {
		ChoppingBoardMemorization freshestMatchingBoardMemorization = null;
		double freshestEpochTime = Double.NEGATIVE_INFINITY;
		for ( ChoppingBoardMemorization board : bot.getObservationMemory().getAllByMemorization( ChoppingBoardMemorization.class ) ) {
			if ( 
				freshestEpochTime < board.getMemorizationEpochTime()
				&&
				bot.getKitchen().isWithin( board.getActorLocation() )
				&&
				( requiredState != BoardState.READY || boardWithPrecursorPredicate.apply( board ) )
				&&
				( requiredState != BoardState.EMPTY || board.readInventory().isEmpty() )
			) {
				freshestMatchingBoardMemorization = board;
				freshestEpochTime = board.getMemorizationEpochTime();
			}
		}
		return freshestMatchingBoardMemorization;
	}
	
	protected final Predicate<IItem> pieceItemPredicate = new Predicate<IItem>() {
		@Override
		public boolean apply( IItem item ) {
			return pieceItemType.getPieceItemClass().isInstance( item );
		}
	};
	
	protected final Predicate<IItem> precursorPredicate = new Predicate<IItem>() {
		@Override
		public boolean apply( IItem item ) {
			return pieceItemType.getPrecursorClass().isInstance( item );
		}
	};
	
	protected final Supplier<Boolean> hasFoundBoardSupplier = new Supplier<Boolean>() {
		@Override
		public Boolean get() {
			return findBoardInMemory( BoardState.BLOCKED ) != null;
		}		
	};
		
	protected final Predicate<IChoppingBoard> boardWithPrecursorPredicate = new Predicate<IChoppingBoard>() {
		@Override
		public boolean apply( IChoppingBoard choppingBoard ) {
			return Collections2.filter( choppingBoard.readInventory(), precursorPredicate ).size() > 0;
		}
	};
	
	protected final Predicate<IItem> kitchenKnifePredicate = new Predicate<IItem>() {
		@Override
		public boolean apply( IItem item ) {
			return item instanceof IKitchenKnife;
		}
	};
		
	protected enum Stage {
		COLLECT_KNIFE,
		COLLECT_PRECURSOR,
		FIND_BOARD,
		CLEAR_BOARD,
		PUT_PRECURSOR_ON_BOARD,
		CHOP_PRECURSOR,
		COLLECT_PIECE_ITEM,
		MERGE,
		DONE
	}
	
	protected enum BoardState {
		 EMPTY,
		 BLOCKED,
		 READY
	}
	
	protected enum CountMethod {
		FIRST_STACK,
		ALL
	}
}
