package cz.cuni.amis.utils.image.compare;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import javax.imageio.ImageIO;

public class ImagePixelCompare {
	
	private File[] files = new File[]{null, null};
	
	private BufferedImage[] images = new BufferedImage[]{null, null};
	
	private Set<Integer>[] colors = new Set[]{ new HashSet<Integer>(), new HashSet<Integer>() };
	
	public ImagePixelCompare(BufferedImage image1, BufferedImage image2) {
		images[0] = image1;
		images[1] = image2;
	}
	
	public ImagePixelCompare(File image1, File image2) {
		files[0] = image1;
		files[1] = image2;
		loadFiles();
	}
	
	

	private void loadFiles() {
		System.out.println("Loading images...");
		for (int i = 0; i < files.length; ++i) {
			if (files[i] != null) {
				if (!files[i].exists()) {
					throw new RuntimeException("File of image " + (i+1) + " (" + files[i].getAbsolutePath() + ") does not exist!");
				}
				try {
					images[i] = ImageIO.read(files[i]);
				} catch (IOException e) {
					throw new RuntimeException("Could not read image from: " + files[i]);
				}
			}
		}
	}

	public void compare() {
		System.out.println("ANALYZING!");
		
		if (files[0] != null) {
			System.out.println("Image 1:            " + files[0].getAbsolutePath());
			System.out.println("Image 2:            " + files[1].getAbsolutePath());
		}
		
		if (images[0] == null) throw new RuntimeException("images[0] is null, initialization failed!");
		if (images[1] == null) throw new RuntimeException("images[1] is null, initialization failed!");
		
		if (images[0].getWidth() != images[1].getWidth()) throw new RuntimeException("images[0].getWidth() == " + images[0].getWidth() + " != " + images[1].getWidth() + " == images[1].getWidth(), images width mismatch!");
		if (images[0].getHeight() != images[1].getHeight()) throw new RuntimeException("images[0].getHeight() == " + images[0].getHeight() + " != " + images[1].getHeight() + " == images[1].getHeight(), images height mismatch!");
		
		int pixels = images[0].getWidth() * images[1].getWidth();
		
		double deltaA = 0;
		double deltaR = 0;
		double deltaG = 0;
		double deltaB = 0;

		for (int x = 0; x < images[0].getWidth(); ++x) {
			for (int y = 0; y < images[0].getHeight(); ++y) {
				int pixel1 = images[0].getRGB(x,y);
				colors[0].add(pixel1);
				int alpha1 = (pixel1 >> 24) & 0xff;
			    int red1 = (pixel1 >> 16) & 0xff;
			    int green1 = (pixel1 >> 8) & 0xff;
			    int blue1 = (pixel1) & 0xff;
			    
			    int pixel2 = images[1].getRGB(x,y);
			    colors[1].add(pixel2);
				int alpha2 = (pixel2 >> 24) & 0xff;
			    int red2 = (pixel2 >> 16) & 0xff;
			    int green2 = (pixel2 >> 8) & 0xff;
			    int blue2 = (pixel2) & 0xff;
			    
			    deltaA += Math.abs(alpha1-alpha2);
			    deltaR += Math.abs(red1-red2);
			    deltaG += Math.abs(green1-green2);
			    deltaB += Math.abs(blue1-blue2);
			}
		}
		
		System.out.println("DONE!");
		
		System.out.println("Image:              " + images[0].getWidth() + "x" + images[0].getWidth() + " = " + pixels + " pixels");
		System.out.println("Image1 colors:      " + colors[0].size());
		System.out.println("Image2 colors:      " + colors[1].size());
		System.out.println("Total delta RED:    " + deltaR);
		System.out.println("Total delta GREEN:  " + deltaG);
		System.out.println("Total delta BLUE:   " + deltaB);
		System.out.println("Avg delta RED:      " + deltaR / pixels);
		System.out.println("Avg delta GREEN:    " + deltaG / pixels);
		System.out.println("Avg delta BLUE:     " + deltaB / pixels);
		
		System.out.println("END!");
	}
	
	
	public static void main(String[] args) {
		if (args.length != 2) {
			printHelp();
			return;
		}
		File file1 = new File(args[0]);
		File file2 = new File(args[1]);
		
		ImagePixelCompare compare = new ImagePixelCompare(file1, file2);
		compare.compare();
	}

	private static void printHelp() {
		System.out.println("===================");
		System.out.println("IMAGE PIXEL COMPARE");
		System.out.println("===================");
		System.out.println("");
		System.out.println("This program computes color distances between two images (separately for every channel)");
		System.out.println("");
		System.out.println("Usage: java -jar image-pixel-compare.jar <path_to_image_1> <path_to_image_2>");
		System.out.println("");
		System.out.println("Example: java -jar image-pixel-compare.jar img1.png img2.jpg");
		System.out.println("");
	}
	
}
